package com.vungle.warren.persistence;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.DatabaseUtils;
import android.database.sqlite.SQLiteDatabase;
import android.os.Environment;
import android.text.TextUtils;
import android.util.Log;

import com.vungle.warren.AdLoader;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.VungleLogger;
import com.vungle.warren.model.AdAsset;
import com.vungle.warren.model.AdAssetDBAdapter;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.AdvertisementDBAdapter;
import com.vungle.warren.model.AnalyticUrl;
import com.vungle.warren.model.AnalyticUrlDBAdapter;
import com.vungle.warren.model.CacheBust;
import com.vungle.warren.model.CacheBustDBAdapter;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.model.CookieDBAdapter;
import com.vungle.warren.model.Placement;
import com.vungle.warren.model.PlacementDBAdapter;
import com.vungle.warren.model.Report;
import com.vungle.warren.model.ReportDBAdapter;
import com.vungle.warren.model.SessionData;
import com.vungle.warren.model.SessionDataDBAdapter;
import com.vungle.warren.model.VisionData;
import com.vungle.warren.model.VisionDataDBAdapter;
import com.vungle.warren.utility.FileUtility;
import com.vungle.warren.utility.VungleThreadPoolExecutor;
import com.vungle.warren.vision.VisionAggregationData;
import com.vungle.warren.vision.VisionAggregationInfo;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import static com.vungle.warren.model.Placement.TYPE_DEFAULT;
import static com.vungle.warren.model.Placement.TYPE_DEPRECATED_TEMPLATE;
import static com.vungle.warren.model.PlacementDBAdapter.PlacementColumns.COLUMN_AD_SIZE;
import static com.vungle.warren.model.PlacementDBAdapter.PlacementColumns.COLUMN_REFRESH_DURATION;
import static com.vungle.warren.model.PlacementDBAdapter.PlacementColumns.COLUMN_SUPPORTED_TEMPLATE_TYPES;
import static com.vungle.warren.model.ReportDBAdapter.ReportColumns.COLUMN_REPORT_STATUS;
import static com.vungle.warren.model.ReportDBAdapter.ReportColumns.COLUMN_TT_DOWNLOAD;

public class Repository {

    private static final String TAG = Repository.class.getSimpleName();

    @VisibleForTesting
    protected DatabaseHelper dbHelper;
    private final VungleThreadPoolExecutor ioExecutor;
    private final ExecutorService uiExecutor;
    private final Designer designer;
    private final Context appCtx;

    @VisibleForTesting
    protected static final int VERSION = 11;

    private Map<Class, DBAdapter> adapters = new HashMap<>();

    public Repository(Context context, Designer designer, VungleThreadPoolExecutor ioExecutor,
                      ExecutorService uiExecutor) {
        this(context, designer, ioExecutor, uiExecutor, VERSION);
    }

    public Repository(Context context, Designer designer, VungleThreadPoolExecutor ioExecutor,
                      ExecutorService uiExecutor, int version) {
        this.appCtx = context.getApplicationContext();
        this.ioExecutor = ioExecutor;
        this.uiExecutor = uiExecutor;

        dbHelper = new DatabaseHelper(context, version, new VungleDatabaseCreator(appCtx));
        this.designer = designer;

        adapters.put(Placement.class, new PlacementDBAdapter());
        adapters.put(Cookie.class, new CookieDBAdapter());
        adapters.put(Report.class, new ReportDBAdapter());
        adapters.put(Advertisement.class, new AdvertisementDBAdapter());
        adapters.put(AdAsset.class, new AdAssetDBAdapter());
        adapters.put(VisionData.class, new VisionDataDBAdapter());
        adapters.put(AnalyticUrl.class, new AnalyticUrlDBAdapter());
        adapters.put(CacheBust.class, new CacheBustDBAdapter());
        adapters.put(SessionData.class, new SessionDataDBAdapter());
    }

    public void init() throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                dbHelper.init();
                ContentValues contentValues = new ContentValues();
                contentValues.put(AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE, Advertisement.DONE);

                Query query = new Query(AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME);
                query.selection = AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE + "=?";
                query.args = new String[]{String.valueOf(Advertisement.VIEWING)};

                dbHelper.update(query, contentValues);
                return null;
            }
        });
    }

    private <T> List<T> loadAllModels(Class<T> tClass) {
        DBAdapter<T> adapter = adapters.get(tClass);
        if (adapter == null) return Collections.EMPTY_LIST;

        Cursor cursor = dbHelper.query(new Query(adapter.tableName()));

        return extractModels(tClass, cursor);
    }

    @NonNull
    private <T> List<T> extractModels(Class<T> clazz, Cursor cursor) {
        if (cursor == null || cursor.isClosed()) {
            return Collections.emptyList();
        }

        List<T> items = new ArrayList<>();

        try {
            DBAdapter<T> adapter = adapters.get(clazz);

            while (cursor.moveToNext()) {
                ContentValues values = new ContentValues();
                DatabaseUtils.cursorRowToContentValues(cursor, values);
                items.add(adapter.fromContentValues(values));
            }
        } catch (Exception e) {
            VungleLogger.critical(
                    true,
                    Repository.class.getSimpleName(),
                    "extractModels",
                    e.toString()
            );
            return new ArrayList<>();
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        return items;
    }

    private <T> T loadModel(String id, Class<T> tClass) {
        DBAdapter<T> adapter = adapters.get(tClass);

        Query query = new Query(adapter.tableName());
        query.selection = IdColumns.COLUMN_IDENTIFIER + " = ? ";
        query.args = new String[]{id};

        Cursor cursor = dbHelper.query(query);

        if (cursor != null) {
            try {
                if (cursor.moveToNext()) {
                    ContentValues values = new ContentValues();
                    DatabaseUtils.cursorRowToContentValues(cursor, values);
                    return adapter.fromContentValues(values);
                }
            } catch (Exception e) {
                VungleLogger.critical(
                        true,
                        Repository.class.getSimpleName(),
                        "loadModel",
                        e.toString()
                );
                return null;
            } finally {
                cursor.close();
            }
        }

        return null;
    }

    private <T> void saveModel(T model) throws DatabaseHelper.DBException {
        DBAdapter<T> adapter = adapters.get(model.getClass());
        ContentValues contentValues = adapter.toContentValues(model);
        dbHelper.insertWithConflict(adapter.tableName(), contentValues, SQLiteDatabase.CONFLICT_REPLACE);
    }

    public <T> FutureResult<T> load(@NonNull final String id, @NonNull final Class<T> clazz) {
        return new FutureResult<>(ioExecutor.submit(new Callable<T>() {
            @Override
            public T call() {
                return loadModel(id, clazz);
            }
        })
        );
    }

    public <T> void load(@NonNull final String id, @NonNull final Class<T> clazz,
                         @NonNull final LoadCallback<T> loadCallback) {
        ioExecutor.execute(new Runnable() {
            @Override
            public void run() {
                final T result = loadModel(id, clazz);
                uiExecutor.execute(new Runnable() {
                    @Override
                    public void run() {
                        loadCallback.onLoaded(result);
                    }
                });
            }
        });
    }

    public <T> void save(final T item) throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                saveModel(item);
                return null;
            }
        });
    }

    /**
     * Saves passed <code>item</code> to repository. Upon completion will call either of callback
     * methods in <code>callback</code>. <code>waitForResult</code> lets caller choose whether this
     * call will be blocking or not. There's also convenience call {@link #save(Object, SaveCallback)}
     * with this parameter set to <code>true</code>
     *
     * @param item          item to be saved in Repository
     * @param callback      callback to be used to signal {@link SaveCallback#onSaved()} or
     *                      {@link SaveCallback#onError(Exception)} depending on the result of this
     *                      save
     * @param waitForResult set <code>true</code> to make this call blocking
     * @param <T>           type of item to be saved
     */
    public <T> void save(final T item, @Nullable final SaveCallback callback, boolean waitForResult) {
        Future<?> future = ioExecutor.submit(new Runnable() {
            @Override
            public void run() {
                try {
                    saveModel(item);
                } catch (final DatabaseHelper.DBException e) {
                    onSaveCallbackError(callback, e);
                    return;
                }
                if (callback != null) {
                    uiExecutor.execute(new Runnable() {
                        @Override
                        public void run() {
                            callback.onSaved();
                        }
                    });
                }
            }
        }, new Runnable() {
            @Override
            public void run() {
                onSaveCallbackError(callback, new VungleException(VungleException.OUT_OF_MEMORY));
            }
        });
        if (waitForResult) {
            try {
                future.get();
            } catch (InterruptedException e) {
                Log.e(TAG, "InterruptedException ", e);
                Thread.currentThread().interrupt();
            } catch (ExecutionException e) {
                Log.e(TAG, "Error on execution during saving", e);
            }
        }
    }

    private void onSaveCallbackError(final SaveCallback callback, final Exception e) {
        if (callback != null) {
            uiExecutor.execute(new Runnable() {
                @Override
                public void run() {
                    callback.onError(e);
                }
            });
        }
    }

    /**
     * Shortcut implemetation of {@link #save(Object, SaveCallback, boolean)} with
     * <code>waitForResult</code> set to <code>true</code>
     *
     * @param item     item to be saved in Repository
     * @param callback callback to be used to signal {@link SaveCallback#onSaved()} or
     *                 {@link SaveCallback#onError(Exception)} depending on the result of this save
     * @param <T>      type of item to be saved
     */
    public <T> void save(final T item, @Nullable final SaveCallback callback) {
        save(item, callback, true);
    }

    /**
     *
     * @param placementId id of placement
     * @param eventId if eventId is non-null, will find advertisement associated by event ID.
     *                if not, will simply use the advertisement solely by placement Id
     *
     *                SQL query looks like. The last clause is conditional based on if
     *                {@param eventId} is null or not.
     *
     *
     *
     *
     *                SELECT * from TABLE "advertisement" WHERE
     *                placement_id = {placementId} AND
     *                (state = "READY" OR state = "NEW") AND
     *                expire_time > {currentSystemTime} AND
     *                item_id = {@param eventId}
     *                ORDER BY state DESC
     *
     * @return list of advertisements where the event ID corresponds. If there is no event ID, use
     * only the placementId.
     */
    private List<Advertisement> findValidAdvertisementsForPlacementFromDB(
            String placementId, String eventId
    ) {
        Log.i(TAG, " Searching for valid advertisement for placement with " + placementId +
                "event ID " + eventId);

        Query query = new Query(AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME);

        StringBuilder sb = new StringBuilder()
                .append(AdvertisementDBAdapter.AdvertisementColumns.COLUMN_PLACEMENT_ID + " = ? AND ")
                .append("(" + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE + " = ? OR ")
                .append(AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE + " = ?) AND ")
                .append(AdvertisementDBAdapter.AdvertisementColumns.COLUMN_EXPIRE_TIME + " > ?");

        String[] args;
        if (eventId != null) {
            sb.append(" AND " + IdColumns.COLUMN_IDENTIFIER + " = ?");
            args = new String[]{
                    placementId,
                    String.valueOf(Advertisement.READY),
                    String.valueOf(Advertisement.NEW),
                    String.valueOf(System.currentTimeMillis() / 1000L),
                    eventId
            };
        } else {
            args = new String[]{
                    placementId,
                    String.valueOf(Advertisement.READY),
                    String.valueOf(Advertisement.NEW),
                    String.valueOf(System.currentTimeMillis() / 1000L)
            };
        }

        query.selection = sb.toString();
        query.args = args;
        query.orderBy = AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE + " DESC";

        AdvertisementDBAdapter adapter = (AdvertisementDBAdapter) adapters.get(Advertisement.class);
        List<Advertisement> items = new ArrayList<>();
        Cursor cursor = dbHelper.query(query);
        if (cursor == null) {
            return items;
        }
        try {
            while (adapter != null && cursor.moveToNext()) {
                ContentValues values = new ContentValues();
                DatabaseUtils.cursorRowToContentValues(cursor, values);
                items.add(adapter.fromContentValues(values));
            }
        } catch (Exception e) {
            VungleLogger.critical(
                    true,
                    Repository.class.getSimpleName(),
                    "findValidAdvertisementsForPlacementFromDB",
                    e.toString()
            );
            return new ArrayList<>();
        } finally {
            cursor.close();
        }
        return items;
    }

    /**
     *
     * @param placementId id of placement
     * @param eventId if eventId is non-null, will find advertisement associated by event ID.
     *                if not, will simply use the advertisement solely by placement Id
     *
     *                SQL query looks like. The last clause is conditional based on if
     *                {@param eventId} is null or not.
     *
     *                SELECT * from TABLE "advertisement" WHERE
     *                placement_id = {placementId} AND
     *                (state = "READY" OR state = "NEW") AND
     *                expire_time > {currentSystemTime} AND
     *                item_id = {eventId}
     *
     * @return Advertisement where the event ID corresponds. If there is no event ID, use
     * only the placementId.
     */
    private @Nullable Advertisement findValidAdvertisementForPlacementFromDB(
            @NonNull String placementId, @Nullable String eventId
    ) {
        Log.i(TAG, " Searching for valid advertisement for placement with " + placementId +
                "event ID " + eventId);

        Query query = new Query(AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME);

        StringBuilder sb = new StringBuilder()
                .append(AdvertisementDBAdapter.AdvertisementColumns.COLUMN_PLACEMENT_ID + " = ? AND ")
                .append("(" + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE + " = ? OR ")
                .append(AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE + " = ?) AND ")
                .append(AdvertisementDBAdapter.AdvertisementColumns.COLUMN_EXPIRE_TIME + " > ?");

        String[] args;
        if (eventId != null) {
            sb.append(" AND " + IdColumns.COLUMN_IDENTIFIER + " = ?");
            args = new String[]{
                    placementId,
                    String.valueOf(Advertisement.READY),
                    String.valueOf(Advertisement.NEW),
                    String.valueOf(System.currentTimeMillis() / 1000L),
                    eventId
            };
        } else {
            args = new String[]{
                    placementId,
                    String.valueOf(Advertisement.READY),
                    String.valueOf(Advertisement.NEW),
                    String.valueOf(System.currentTimeMillis() / 1000L)
            };
        }

        query.selection = sb.toString();

        query.args = args;
        query.limit = "1";
        Advertisement advertisement = null;
        Cursor cursor = dbHelper.query(query);
        if (cursor == null) {
            return null;
        }
        try {
            AdvertisementDBAdapter adapter = (AdvertisementDBAdapter) adapters.get(Advertisement.class);
            if (adapter != null && cursor.moveToNext()) {
                ContentValues values = new ContentValues();
                DatabaseUtils.cursorRowToContentValues(cursor, values);
                advertisement = adapter.fromContentValues(values);
            }
        } catch (Exception e) {
            VungleLogger.critical(
                    true,
                    Repository.class.getSimpleName(),
                    "findValidAdvertisementForPlacementFromDB",
                    e.toString()
            );
            return null;
        } finally {
            cursor.close();
        }
        return advertisement;
    }

    /**
     * Looks for valid advertisement for {@link Placement} id.
     *
     * @param placementId
     * @param eventId
     * @return non expired {@link Advertisement} with {@link Advertisement#READY}
     * or {@link Advertisement#NEW} states
     */
    public FutureResult<Advertisement> findValidAdvertisementForPlacement(final String placementId, @Nullable final String eventId) {
        return new FutureResult<>(ioExecutor.submit(new Callable<Advertisement>() {
            @Override
            public Advertisement call() {
                return findValidAdvertisementForPlacementFromDB(placementId, eventId);
            }
        }));
    }

    /**
     * Searches if there are any ad that has expired.
     *
     * @return <code>true</code> if there is one or more expired ad for a single placement. Supports
     * the Multiple Header Bidding case.
     *
     * SELECT * from TABLE "advertisement" WHERE
     * placement_id = {placementId} AND
     * (state = "READY" OR state = "NEW") AND
     * item_id = {@param eventId} // when there is an eventId.
     **/
    public FutureResult<Advertisement> findPotentiallyExpiredAd(final String placementId, final String eventId) {
        Log.i(TAG, " Searching for valid advertisement for placement with " + placementId +
                " event ID " + eventId);
        return new FutureResult<>(ioExecutor.submit(new Callable<Advertisement>() {
            @Override
            public Advertisement call() {
                Query query = new Query(AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME);

                StringBuilder sb = new StringBuilder()
                        .append(AdvertisementDBAdapter.AdvertisementColumns.COLUMN_PLACEMENT_ID + " = ? AND ")
                        .append("(" + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE + " = ? OR ")
                        .append(AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE + " = ?)");

                String[] args;
                if (eventId != null) {
                    sb.append(" AND " + IdColumns.COLUMN_IDENTIFIER + " = ?");
                    args = new String[]{
                            placementId,
                            String.valueOf(Advertisement.READY),
                            String.valueOf(Advertisement.NEW),
                            eventId
                    };
                } else {
                    args = new String[]{
                            placementId,
                            String.valueOf(Advertisement.READY),
                            String.valueOf(Advertisement.NEW),
                    };
                }

                query.selection = sb.toString();
                query.args = args;

                Advertisement advertisement = null;
                Cursor cursor = dbHelper.query(query);
                if (cursor == null) {
                    return null;
                }
                try {
                    AdvertisementDBAdapter adapter =
                            (AdvertisementDBAdapter) adapters.get(Advertisement.class);
                    if (adapter != null && cursor.moveToNext()) {
                        ContentValues values = new ContentValues();
                        DatabaseUtils.cursorRowToContentValues(cursor, values);
                        advertisement = adapter.fromContentValues(values);
                    }
                } catch (Exception e) {
                    VungleLogger.critical(
                            true,
                            Repository.class.getSimpleName(),
                            "findPotentiallyExpiredAd",
                            e.toString()
                    );
                    return null;
                } finally {
                    cursor.close();
                }
                return advertisement;
            }
        }));
    }

    /**
     * Looks for valid advertisement for {@link Placement} id.
     *
     * @param placementId
     * @param eventId
     * @return list of non expired {@link Advertisement} with {@link Advertisement#READY}
     * or {@link Advertisement#NEW} states
     */
    public FutureResult<List<Advertisement>> findValidAdvertisementsForPlacement(
            final String placementId, @Nullable final String eventId
    ) {
        return new FutureResult<>(ioExecutor.submit(new Callable<List<Advertisement>>() {
            @Override
            public List<Advertisement> call() {
                return findValidAdvertisementsForPlacementFromDB(placementId, eventId);
            }
        }));
    }

    public <T> FutureResult<List<T>> loadAll(final Class<T> clazz) {

        return new FutureResult<>(ioExecutor.submit(new Callable<List<T>>() {
            @Override
            public List<T> call() {
                return loadAllModels(clazz);
            }
        }));
    }

    public @Nullable
    FutureResult<List<Report>> loadAllReportToSend() {
        return new FutureResult<>(ioExecutor.submit(new Callable<List<Report>>() {
            @Override
            public List<Report> call() {
                List<Report> sendReports = loadAllModels(Report.class);
                for (Report report : sendReports) {
                    report.setStatus(Report.SENDING);
                    try {
                        saveModel(report);
                    } catch (DatabaseHelper.DBException e) {
                        return null;
                    }
                }
                return sendReports;
            }
        }));
    }

    public @Nullable
    FutureResult<List<Report>> loadReadyOrFailedReportToSend() {
        //First set any existing report to Sending status whose status is Ready/Failed, then update it to Sending

        return new FutureResult<>(ioExecutor.submit(new Callable<List<Report>>() {
            @Override
            public List<Report> call() {
                Query query = new Query(ReportDBAdapter.ReportColumns.TABLE_NAME);
                query.selection =
                        COLUMN_REPORT_STATUS + " = ? " + " OR " +
                                COLUMN_REPORT_STATUS + " = ? ";
                query.args = new String[]{String.valueOf(Report.READY), String.valueOf(Report.FAILED)};
                final Cursor cursor = dbHelper.query(query);
                List<Report> sendReports = extractModels(Report.class, cursor);
                for (Report report : sendReports) {
                    report.setStatus(Report.SENDING);
                    try {
                        saveModel(report);
                    } catch (DatabaseHelper.DBException e) {
                        return null;
                    }
                }
                return sendReports;
            }
        }));
    }

    public void updateAndSaveReportState(final String placementId, final String appId, final int statusFrom, final int statusTo) throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                ContentValues contentValues = new ContentValues();
                contentValues.put(COLUMN_REPORT_STATUS, statusTo);

                Query query = new Query(ReportDBAdapter.ReportColumns.TABLE_NAME);
                query.selection = ReportDBAdapter.ReportColumns.COLUMN_PLACEMENT_ID + " = ? " + " AND " +
                        COLUMN_REPORT_STATUS + " = ? " + " AND " +
                        ReportDBAdapter.ReportColumns.COLUMN_APP_ID + " = ? ";
                query.args = new String[]{placementId, String.valueOf(statusFrom), appId};

                dbHelper.update(query, contentValues);
                return null;
            }
        });
    }

    public FutureResult<List<AdAsset>> loadAllAdAssets(@NonNull final String adId) {
        return new FutureResult<>(ioExecutor.submit(new Callable<List<AdAsset>>() {
            @Override
            public List<AdAsset> call() {
                return loadAllAdAssetModels(adId);
            }
        }));
    }

    private List<AdAsset> loadAllAdAssetModels(@NonNull String adId) {
        Query query = new Query(AdAssetDBAdapter.AdAssetColumns.TABLE_NAME);
        query.selection = AdAssetDBAdapter.AdAssetColumns.COLUMN_AD_ID + " = ? ";
        query.args = new String[]{adId};
        Cursor cursor = dbHelper.query(query);
        return extractModels(AdAsset.class, cursor);
    }

    public List<AdAsset> loadAllAdAssetByStatus(@NonNull String adId, @AdAsset.Status final int status) {
        Query query = new Query(AdAssetDBAdapter.AdAssetColumns.TABLE_NAME);
        query.selection = AdAssetDBAdapter.AdAssetColumns.COLUMN_AD_ID + " = ? "+ " AND " +
                AdAssetDBAdapter.AdAssetColumns.COLUMN_FILE_STATUS + " = ? ";
        query.args = new String[]{adId, String.valueOf(status)};
        Cursor cursor = dbHelper.query(query);
        return extractModels(AdAsset.class, cursor);
    }

    public <T> void delete(final T r) throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                deleteModel(r);
                return null;
            }
        });
    }

    private <T> void deleteModel(Class<T> clazz, String id) throws DatabaseHelper.DBException {
        DBAdapter<T> adapter = adapters.get(clazz);
        Query query = new Query(adapter.tableName());
        query.selection = IdColumns.COLUMN_IDENTIFIER + "=?";
        query.args = new String[]{id};
        dbHelper.delete(query);
    }

    private void deleteAssetForAdId(String adId) throws DatabaseHelper.DBException {
        DBAdapter adapter = adapters.get(AdAsset.class);
        Query query = new Query(adapter.tableName());
        query.selection = AdAssetDBAdapter.AdAssetColumns.COLUMN_AD_ID + "=?";
        query.args = new String[]{adId};
        dbHelper.delete(query);
    }

    private <T> void deleteModel(T model) throws DatabaseHelper.DBException {
        DBAdapter<T> adapter = adapters.get(model.getClass());
        ContentValues contentValues = adapter.toContentValues(model);
        deleteModel(model.getClass(), contentValues.getAsString(IdColumns.COLUMN_IDENTIFIER));
    }

    public void deleteAdvertisement(final String advertisementId) throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                deleteAdInternal(advertisementId);
                return null;
            }
        });
    }

    private void deleteAdInternal(String advertisementId) throws DatabaseHelper.DBException {
        if (TextUtils.isEmpty(advertisementId))
            return;

        //First deleting all Ad Assets before deleting the AD
        deleteAssetForAdId(advertisementId);
        deleteModel(Advertisement.class, advertisementId);

        try {
            designer.deleteAssets(advertisementId);
        } catch (IOException e) {
            Log.e(TAG, "IOException ", e);
        }
    }

    /**
     * Makes copy and returns currently valid placements
     *
     * @return currently valid {@link Placement} list
     */
    public FutureResult<Collection<Placement>> loadValidPlacements() {
        return new FutureResult<>(ioExecutor.submit(new Callable<Collection<Placement>>() {
            @Override
            public List<Placement> call() {
                synchronized (Repository.this) {
                    Query query = new Query(PlacementDBAdapter.PlacementColumns.TABLE_NAME);
                    query.selection = PlacementDBAdapter.PlacementColumns.COLUMN_IS_VALID + " = ?";
                    query.args = new String[]{"1"};

                    Cursor cursor = dbHelper.query(query);

                    return extractModels(Placement.class, cursor);
                }
            }
        }));
    }

    private List<String> loadValidPlacementIds() {
        Query query = new Query(PlacementDBAdapter.PlacementColumns.TABLE_NAME);
        query.selection = PlacementDBAdapter.PlacementColumns.COLUMN_IS_VALID + " = ?";
        query.args = new String[]{"1"};
        query.columns = new String[]{PlacementDBAdapter.PlacementColumns.COLUMN_IDENTIFIER};
        Cursor cursor = dbHelper.query(query);
        List<String> ids = new ArrayList<>();

        if (cursor != null) {
            try {
                while (cursor != null && cursor.moveToNext()) {
                    ids.add(cursor.getString(cursor.getColumnIndex(PlacementDBAdapter.PlacementColumns.COLUMN_IDENTIFIER)));
                }
            } catch (Exception e) {
                VungleLogger.critical(
                        true,
                        Repository.class.getSimpleName(),
                        "loadValidPlacementIds",
                        e.toString()
                );
            } finally {
                cursor.close();
            }
        }

        return ids;
    }

    public FutureResult<File> getAdvertisementAssetDirectory(final String id) {
        return new FutureResult<>(ioExecutor.submit(new Callable<File>() {
            @Override
            public File call() throws Exception {
                return designer.getAssetDirectory(id);
            }
        }));
    }

    /**
     * Makes copy and returns currently valid placements
     *
     * @return currently valid {@link Placement} list
     */
    public FutureResult<Collection<String>> getValidPlacementIds() {
        return new FutureResult<>(ioExecutor.submit(new Callable<Collection<String>>() {
            @Override
            public Collection<String> call() throws Exception {
                synchronized (Repository.this) {
                    return loadValidPlacementIds();
                }
            }
        })
        );
    }

    /**
     * Makes copy and returns currently available bid tokens.
     *
     * @param maxNumberOfBytes the bid tokens size limitation.
     * @param commaDelimiterBytePadding extra allocation of size for string for any padding that will be added
     * @return currently available {@link Advertisement} list which contains bid token.
     */
    public FutureResult<List<String>> getAvailableBidTokens(
            @Nullable final String placementId,
            final int maxNumberOfBytes,
            final int commaDelimiterBytePadding
    ) {
        return new FutureResult<>(ioExecutor.submit(new Callable<List<String>>() {
            @Override
            public List<String> call() {
                synchronized (Repository.this) {
                    Query query = new Query(AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME);

                    String selection = AdvertisementDBAdapter.AdvertisementColumns.COLUMN_BID_TOKEN + " != ''" +
                            " AND ( " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE + " = ?" +
                            " OR " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_STATE + " = ?" +
                            " ) AND " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_EXPIRE_TIME + " > ?";

                    if (!TextUtils.isEmpty(placementId)) {
                        selection += " AND " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_PLACEMENT_ID + " = ?";
                    }

                    query.selection = selection;
                    query.columns = new String[]{AdvertisementDBAdapter.AdvertisementColumns.COLUMN_BID_TOKEN};

                    String[] arguments = new String[] {
                            String.valueOf(Advertisement.NEW),
                            String.valueOf(Advertisement.READY),
                            String.valueOf(System.currentTimeMillis() / 1000L)
                    };
                    if (!TextUtils.isEmpty(placementId)) {
                        arguments = new String[] {
                                String.valueOf(Advertisement.NEW),
                                String.valueOf(Advertisement.READY),
                                String.valueOf(System.currentTimeMillis() / 1000L),
                                placementId
                        };
                    }

                    query.args = arguments;
                    Cursor cursor = dbHelper.query(query);
                    List<String> bidTokens = new ArrayList<>();
                    int sizeOfTokens = 0;

                    if (cursor != null) {
                        try {
                            while (cursor.moveToNext() && (sizeOfTokens < maxNumberOfBytes)) {
                                int bidTokenColumnIndex = cursor.getColumnIndex(AdvertisementDBAdapter.AdvertisementColumns.COLUMN_BID_TOKEN);
                                String bidToken = cursor.getString(bidTokenColumnIndex);

                                if (sizeOfTokens + bidToken.getBytes().length <= maxNumberOfBytes) {
                                    sizeOfTokens += (bidToken.getBytes().length + commaDelimiterBytePadding);
                                    bidTokens.add(bidToken);
                                }
                            }
                        } catch (Exception e) {
                            VungleLogger.critical(
                                    true,
                                    Repository.class.getSimpleName(),
                                    "getAvailableBidTokens",
                                    e.toString()
                            );
                            return new ArrayList<>();
                        } finally {
                            cursor.close();
                        }
                    }
                    return bidTokens;
                }
            }
        })
        );
    }

    /**
     * Sets currently valid placement objects.
     * Verifies current placement and stored.
     * Deletes assets for non-valid placements
     *
     * @param placements to save
     */
    public void setValidPlacements(final @NonNull List<Placement> placements) throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                synchronized (Repository.class) {
                    ContentValues contentValues = new ContentValues();
                    contentValues.put(PlacementDBAdapter.PlacementColumns.COLUMN_IS_VALID, false);

                    dbHelper.update(new Query(PlacementDBAdapter.PlacementColumns.TABLE_NAME), contentValues);

                    for (Placement placement : placements) {
                        Placement disk = loadModel(placement.getId(), Placement.class);

                        if (disk != null &&
                                (disk.isIncentivized() != placement.isIncentivized() ||
                                        disk.isHeaderBidding() != placement.isHeaderBidding())) {
                            /// if there was a placement on disk but is not equal to the one from the
                            /// server, overwrite it with the new data. In this case, we also delete any
                            /// assets we have for this placement.
                            Log.w(TAG, "Placements data for " + placement.getId()
                                    + " is different from disc, deleting old");
                            List<String> adIds = getAdsForPlacement(placement.getId());

                            for (String id : adIds) {
                                deleteAdInternal(id);
                            }

                            deleteModel(Placement.class, disk.getId());

                        }

                        //keep non-server values
                        if (disk != null) {
                            placement.setWakeupTime(disk.getWakeupTime());
                            placement.setAdSize(disk.getAdSize());
                        }

                        //For any Placement with deprecated template types, we keep them invalid so theres no caching
                        placement.setValid(placement.getPlacementAdType() != TYPE_DEPRECATED_TEMPLATE);

                        if (placement.getMaxHbCache() == Placement.INVALID_INTEGER_VALUE) {
                            placement.setValid(false);
                        }

                        saveModel(placement);
                    }
                }
                return null;
            }
        });
    }

    public FutureResult<List<String>> findAdsForPlacement(final String placementId) {
        return new FutureResult<>(ioExecutor.submit(new Callable<List<String>>() {
            @Override
            public List<String> call() {
                return getAdsForPlacement(placementId);
            }
        }));
    }

    private List<String> getAdsForPlacement(String id) {
        Query query = new Query(AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME);
        query.columns = new String[]{AdvertisementDBAdapter.AdvertisementColumns.COLUMN_IDENTIFIER};
        query.selection = AdvertisementDBAdapter.AdvertisementColumns.COLUMN_PLACEMENT_ID + "=?";
        query.args = new String[]{id};

        List<String> ids = new ArrayList<>();
        Cursor cursor = dbHelper.query(query);
        if (cursor == null) {
            return ids;
        }
        try {
            while (cursor.moveToNext()) {
                ids.add(cursor.getString(cursor.getColumnIndex(AdvertisementDBAdapter.AdvertisementColumns.COLUMN_IDENTIFIER)));
            }
        } catch (Exception e) {
            VungleLogger.critical(
                    true,
                    Repository.class.getSimpleName(),
                    "getAdsForPlacement",
                    e.toString()
            );
            return new ArrayList<>();
        } finally {
            cursor.close();
        }

        return ids;
    }

    /**
     * @return {@link Advertisement}(s) that have one of provided <code>creativeIds</code> as creative ID
     */
    public List<Advertisement> getAdsByCreative(Collection<String> creativeIds) {
        Set<String> idsSet = new HashSet<>(creativeIds);
        Collection<Advertisement> advertisements = new HashSet<>();
        List<Advertisement> allModels = loadAllModels(Advertisement.class);
        for (Advertisement advertisement : allModels) {
            if (idsSet.contains(advertisement.getCreativeId())) {
                advertisements.add(advertisement);
            }
        }
        return new ArrayList<>(advertisements);
    }

    public List<Advertisement> getAdsByCreative(final String creativeId) {
        return getAdsByCreative(Collections.singletonList(creativeId));
    }


    /**
     * @return {@link Advertisement}(s) that have one of provided <code>campaignIds</code> as campaign ID
     */
    public List<Advertisement> getAdsByCampaign(Collection<String> campaignIds) {
        Set<String> idsSet = new HashSet<>(campaignIds);
        Collection<Advertisement> advertisements = new HashSet<>();
        List<Advertisement> allModels = loadAllModels(Advertisement.class);
        for (Advertisement advertisement : allModels) {
            if (idsSet.contains(advertisement.getCampaignId())) {
                advertisements.add(advertisement);
            }
        }
        return new ArrayList<>(advertisements);
    }

    public List<Advertisement> getAdsByCampaign(final String campaignId) {
        return getAdsByCampaign(Collections.singletonList(campaignId));
    }

    /**
     * @return <code>placementId</code> of provided {@link Advertisement}
     */
    public String getPlacementIdByAd(Advertisement advertisement) {
        return advertisement.getPlacementId();
    }

    public List<CacheBust> getUnProcessedBusts() {
        List<CacheBust> cacheBusts = loadAllModels(CacheBust.class);
        List<CacheBust> unProcessedBusts = new ArrayList<>();

        for (CacheBust bust : cacheBusts) {
            if (bust.getTimestampProcessed() == 0) {
                unProcessedBusts.add(bust);
            }
        }
        return unProcessedBusts;
    }

    public void clearAllData() {
        dbHelper.dropDb();
        designer.clearCache();
    }

    public void close() {
        dbHelper.close();
    }

    public interface LoadCallback<T> {
        void onLoaded(T result);
    }

    public interface SaveCallback {
        void onSaved();

        void onError(Exception e);
    }

    /**
     * Processes both {@param advertisement} and {@param placement}
     * in appropriate states.
     * {@link Advertisement#NEW} - Saves advertisement and assigns to placement
     * {@link Advertisement#READY} - Saves advertisement in ready state
     * {@link Advertisement#VIEWING} - Saves advertisement removes from placement
     * {@link Advertisement#DONE} - Deletes advertisement, its assets and removes from placement
     * In all cases placement is saved
     *
     * @param advertisement {@link Advertisement} to process
     * @param placementId   Associated {@link Placement} {@link String} Id
     * @param state         {@link Advertisement.State} to process
     * @throws DatabaseHelper.DBException in case of database error(s)
     */
    //Move to Vungle class
    public void saveAndApplyState(@NonNull final Advertisement advertisement,
                                  @NonNull final String placementId,
                                  @Advertisement.State final int state) throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                Log.i(TAG, "Setting " + state + " for adv " + advertisement.getId()
                        + " and pl " + placementId);

                advertisement.setState(state);

                switch (state) {
                    case Advertisement.INVALID:
                    case Advertisement.NEW:
                    case Advertisement.READY:
                        advertisement.setPlacementId(placementId);
                        saveModel(advertisement);
                        //No op
                        break;

                    case Advertisement.VIEWING:
                        advertisement.setPlacementId(null);
                        saveModel(advertisement);
                        break;

                    case Advertisement.DONE:
                    case Advertisement.ERROR:
                        deleteAdInternal(advertisement.getId());
                        break;
                }
                return null;
            }
        });
    }

    /**
     * DELETE FROM vision_data WHERE _id <= (SELECT MAX(_id) FROM vision_data) - {@param size}
     */
    public void trimVisionData(final int size) throws DatabaseHelper.DBException {
        runAndWait(new Callable<Void>() {
            @Override
            public Void call() throws Exception {
                Query query = new Query(VisionDataDBAdapter.VisionDataColumns.TABLE_NAME);
                query.selection = VisionDataDBAdapter.VisionDataColumns._ID
                        + " <= ( SELECT MAX( " + VisionDataDBAdapter.VisionDataColumns._ID + " ) FROM "
                        + VisionDataDBAdapter.VisionDataColumns.TABLE_NAME + " ) - ?";
                query.args = new String[]{Integer.toString(size)};
                dbHelper.delete(query);
                return null;
            }
        });
    }

    /**
     * SELECT * FROM vision_data WHERE timestamp >= {@param after} ORDER BY _id DESC
     */
    public FutureResult<VisionAggregationInfo> getVisionAggregationInfo(final long after) {
        return new FutureResult<>(ioExecutor.submit(new Callable<VisionAggregationInfo>() {
            @Override
            public VisionAggregationInfo call() {
                Query query = new Query(VisionDataDBAdapter.VisionDataColumns.TABLE_NAME);
                query.selection = VisionDataDBAdapter.VisionDataColumns.COLUMN_TIMESTAMP + " >= ?";
                query.orderBy = VisionDataDBAdapter.VisionDataColumns._ID + " DESC";
                query.args = new String[]{Long.toString(after)};
                Cursor cursor = dbHelper.query(query);
                VisionDataDBAdapter adapter = (VisionDataDBAdapter) adapters.get(VisionData.class);
                if (cursor != null) {
                    try {
                        if (adapter != null && cursor.moveToFirst()) {
                            ContentValues values = new ContentValues();
                            DatabaseUtils.cursorRowToContentValues(cursor, values);
                            VisionData data = adapter.fromContentValues(values);
                            return new VisionAggregationInfo(cursor.getCount(), data.creative);
                        }
                    } catch (Exception e) {
                        VungleLogger.critical(
                                true,
                                Repository.class.getSimpleName(),
                                "getVisionAggregationInfo",
                                e.toString()
                        );
                        return null;
                    } finally {
                        cursor.close();
                    }
                }
                return null;
            }
        }));
    }

    /**
     * SELECT COUNT(*) as viewCount, MAX(timestamp), {@param filter} FROM vision_data WHERE timestamp >= {@param after} GROUP By {@param filter} ORDER BY _id DESC limit {@param limit}
     */
    public FutureResult<List<VisionAggregationData>> getVisionAggregationData(final long after, final int limit, final String filter) {
        return new FutureResult<>(ioExecutor.submit(new Callable<List<VisionAggregationData>>() {
            @Override
            public List<VisionAggregationData> call() {
                List<VisionAggregationData> list = new ArrayList<>();
                if (!VisionDataDBAdapter.VisionDataColumns.COLUMN_ADVERTISER.equals(filter)
                        && !VisionDataDBAdapter.VisionDataColumns.COLUMN_CAMPAIGN.equals(filter)
                        && !VisionDataDBAdapter.VisionDataColumns.COLUMN_CREATIVE.equals(filter))
                    return list;
                final String viewCount = "viewCount";
                final String lastTimeStamp = "lastTimeStamp";
                Query query = new Query(VisionDataDBAdapter.VisionDataColumns.TABLE_NAME);
                query.columns = new String[]{
                        "COUNT ( * ) as " + viewCount,
                        "MAX ( " + VisionDataDBAdapter.VisionDataColumns.COLUMN_TIMESTAMP + " ) as " + lastTimeStamp,
                        filter};
                query.selection = VisionDataDBAdapter.VisionDataColumns.COLUMN_TIMESTAMP + " >= ?";
                query.groupBy = filter;
                query.orderBy = VisionDataDBAdapter.VisionDataColumns._ID + " DESC";
                query.limit = Integer.toString(limit);
                query.args = new String[]{Long.toString(after)};
                Cursor cursor = dbHelper.query(query);
                if (cursor != null) {
                    try {
                        while (cursor.moveToNext()) {
                            ContentValues values = new ContentValues();
                            DatabaseUtils.cursorRowToContentValues(cursor, values);
                            list.add(new VisionAggregationData(
                                    values.getAsString(filter),
                                    values.getAsInteger(viewCount),
                                    values.getAsLong(lastTimeStamp)));
                        }
                    } catch (Exception e) {
                        VungleLogger.critical(
                                true,
                                Repository.class.getSimpleName(),
                                "getVisionAggregationInfo",
                                e.toString()
                        );
                        return new ArrayList<>();
                    } finally {
                        cursor.close();
                    }
                }
                return list;
            }
        }));
    }

    private static class VungleDatabaseCreator implements DatabaseHelper.DatabaseFactory {

        private final Context context;

        public VungleDatabaseCreator(Context context) {
            this.context = context;
        }

        @Override
        public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
            //report table had status field added
            if (oldVersion < 2) {
                db.execSQL("ALTER TABLE " + ReportDBAdapter.ReportColumns.TABLE_NAME +
                        " ADD COLUMN " + COLUMN_REPORT_STATUS + " INTEGER DEFAULT " + Report.READY);
            }

            if (oldVersion < 3) {
                db.execSQL(VisionDataDBAdapter.CREATE_VISION_TABLE_QUERY);

                db.execSQL("ALTER TABLE " + ReportDBAdapter.ReportColumns.TABLE_NAME +
                        " ADD COLUMN " + ReportDBAdapter.ReportColumns.COLUMN_AD_SIZE + " TEXT ");

                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + COLUMN_TT_DOWNLOAD + " NUMERIC DEFAULT " + -1);

                db.execSQL("ALTER TABLE " + PlacementDBAdapter.PlacementColumns.TABLE_NAME +
                        " ADD COLUMN " + COLUMN_AD_SIZE + " TEXT ");
                db.execSQL("ALTER TABLE " + PlacementDBAdapter.PlacementColumns.TABLE_NAME +
                        " ADD COLUMN " + COLUMN_REFRESH_DURATION + " NUMERIC DEFAULT " + 0);
                db.execSQL("ALTER TABLE " + PlacementDBAdapter.PlacementColumns.TABLE_NAME +
                        " ADD COLUMN " + COLUMN_SUPPORTED_TEMPLATE_TYPES + " NUMERIC DEFAULT " + TYPE_DEFAULT);
            }

            if (oldVersion < 4) {
                db.execSQL("ALTER TABLE " + PlacementDBAdapter.PlacementColumns.TABLE_NAME +
                        " ADD COLUMN " + PlacementDBAdapter.PlacementColumns.COLUMN_HEADERBIDDING + " SHORT ");
                db.execSQL("ALTER TABLE " + ReportDBAdapter.ReportColumns.TABLE_NAME +
                        " ADD COLUMN " + ReportDBAdapter.ReportColumns.COLUMN_HEADERBIDDING + " SHORT ");
            }

            if (oldVersion < 5) {
                db.execSQL("ALTER TABLE " + PlacementDBAdapter.PlacementColumns.TABLE_NAME +
                        " ADD COLUMN " + PlacementDBAdapter.PlacementColumns.COLUMN_AUTOCACHE_PRIORITY + " NUMERIC DEFAULT " + AdLoader.Priority.LOWEST);

                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_ASSET_DOWNLOAD_TIMESTAMP + " NUMERIC DEFAULT " + 0);

                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_ASSET_DOWNLOAD_DURATION + " NUMERIC DEFAULT " + 0);

                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_AD_REQUEST_START_TIMESTAMP + " NUMERIC DEFAULT " + 0);

                db.execSQL("ALTER TABLE " + ReportDBAdapter.ReportColumns.TABLE_NAME +
                        " ADD COLUMN " + ReportDBAdapter.ReportColumns.COLUMN_INIT_TIMESTAMP + " NUMERIC DEFAULT " + 0);

                db.execSQL("ALTER TABLE " + ReportDBAdapter.ReportColumns.TABLE_NAME +
                        " ADD COLUMN " + ReportDBAdapter.ReportColumns.COLUMN_ASSET_DOWNLOAD_DURATION + " NUMERIC DEFAULT " + 0);
            }

            if (oldVersion < 6) {
                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_ENABLE_OM_SDK + " NUMERIC DEFAULT " + 0);

                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_OM_SDK_EXTRA_VAST + " TEXT ");
            }

            if (oldVersion < 7) {
                db.execSQL(AnalyticUrlDBAdapter.CREATE_ANALYTICS_URL_TABLE_QUERY);
            }

            if (oldVersion < 8) {
                db.execSQL(CacheBustDBAdapter.CREATE_CACHE_BUST_TABLE_QUERY);

                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_SERVER_REQUEST_TIMESTAMP + " NUMERIC DEFAULT " + 0);

                db.execSQL("ALTER TABLE " + PlacementDBAdapter.PlacementColumns.TABLE_NAME +
                        " ADD COLUMN " + PlacementDBAdapter.PlacementColumns.COLUMN_MAX_HB_CACHE + " NUMERIC DEFAULT " + 0);

                db.execSQL("ALTER TABLE " + PlacementDBAdapter.PlacementColumns.TABLE_NAME +
                        " ADD COLUMN " + PlacementDBAdapter.PlacementColumns.COLUMN_RECOMMENDED_AD_SIZE + " TEXT ");

                /*
                update for Advertisement table is done with full drop at
                com.vungle.warren.persistence.DatabaseHelper.NoBackupDatabaseWrapperContext.getDatabasePath
                during upgrade 6.10.0 -> 6.10.1 and subsequent
                com.vungle.warren.persistence.Repository.VungleDatabaseCreator.create
                */
            }
            if (oldVersion < 9) {
                db.execSQL("ALTER TABLE " + ReportDBAdapter.ReportColumns.TABLE_NAME +
                        " ADD COLUMN " + ReportDBAdapter.ReportColumns.COLUMN_PLAY_REMOTE_URL + " SHORT DEFAULT " + 0);
                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_ASSETS_FULLY_DOWNLOADED + " SHORT DEFAULT " + 0);
            }

            if (oldVersion < 10) {
                // 6.11.x Tencent click coordinates collection feature.
                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_CLICK_COORDINATES_ENABLED + " SHORT DEFAULT " + 0);
                // 6.11.x deep link feature
                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_DEEP_LINK + " TEXT ");
                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_NOTIFICATIONS  + " TEXT ");
            }

            if (oldVersion < 11) {
                db.execSQL("ALTER TABLE " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME +
                        " ADD COLUMN " + AdvertisementDBAdapter.AdvertisementColumns.COLUMN_HEADER_BIDDING + " SHORT DEFAULT " + 0);
            }

        }

        @Override
        public void create(SQLiteDatabase db) {
            dropOldFilesData();

            db.execSQL(AdvertisementDBAdapter.CREATE_ADVERTISEMENT_TABLE_QUERY);
            db.execSQL(PlacementDBAdapter.CREATE_PLACEMENT_TABLE_QUERY);
            db.execSQL(CookieDBAdapter.CREATE_COOKIE_TABLE_QUERY);
            db.execSQL(ReportDBAdapter.CREATE_REPORT_TABLE_QUERY);
            db.execSQL(AdAssetDBAdapter.CREATE_ASSET_TABLE_QUERY);
            db.execSQL(VisionDataDBAdapter.CREATE_VISION_TABLE_QUERY);
            db.execSQL(AnalyticUrlDBAdapter.CREATE_ANALYTICS_URL_TABLE_QUERY);
            db.execSQL(CacheBustDBAdapter.CREATE_CACHE_BUST_TABLE_QUERY);
            db.execSQL(SessionDataDBAdapter.CREATE_SESSION_DATA_TABLE_QUERY);
        }

        @Override
        public void deleteData(SQLiteDatabase db) {
            db.execSQL("DROP TABLE IF EXISTS " + AdvertisementDBAdapter.AdvertisementColumns.TABLE_NAME);
            db.execSQL("DROP TABLE IF EXISTS " + CookieDBAdapter.CookieColumns.TABLE_NAME);
            db.execSQL("DROP TABLE IF EXISTS " + PlacementDBAdapter.PlacementColumns.TABLE_NAME);
            db.execSQL("DROP TABLE IF EXISTS " + ReportDBAdapter.ReportColumns.TABLE_NAME);
            db.execSQL("DROP TABLE IF EXISTS " + AdAssetDBAdapter.AdAssetColumns.TABLE_NAME);
            db.execSQL("DROP TABLE IF EXISTS " + VisionDataDBAdapter.VisionDataColumns.TABLE_NAME);
            db.execSQL("DROP TABLE IF EXISTS " + AnalyticUrlDBAdapter.AnalyticsUrlColumns.TABLE_NAME);
            db.execSQL("DROP TABLE IF EXISTS " + CacheBustDBAdapter.CacheBustColumns.TABLE_NAME);
            db.execSQL("DROP TABLE IF EXISTS " + SessionDataDBAdapter.SessionDataColumns.TABLE_NAME);
        }

        @Override
        public void onDowngrade(SQLiteDatabase db, int oldVersion, int newVersion) {
            List<String> names = new ArrayList<>();
            Cursor cursor = db.rawQuery("SELECT * FROM sqlite_master WHERE type='table'", null);
            while (cursor != null && cursor.moveToNext()) {
                String tableName = cursor.getString(1);
                if (!tableName.equals("android_metadata") && !tableName.startsWith("sqlite_")) {
                    names.add(tableName);
                }
            }
            if (cursor != null) {
                cursor.close();
            }
            for (String name : names) {
                db.execSQL("DROP TABLE IF EXISTS " + name);
            }

            create(db);
        }

        private void deleteDatabase(String dbName) {
            context.deleteDatabase(dbName);
        }

        private void dropOldFilesData() {
            //Delete old database if exist for v5.2.x
            deleteDatabase("vungle");                       //'vungle' is the db name of SDK v5.3.x

            //Deleting external dir data for v5.2.x
            final File external = context.getExternalFilesDir(null);
            boolean canUseExternal =
                    Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)
                            && external != null;
            if (canUseExternal && external.exists()) {
                File oldData = new File(external, ".vungle");       //'.vungle' is the assets directory name of SDK v5.3.x
                try {
                    FileUtility.delete(oldData);
                } catch (IOException e) {
                    Log.e(TAG, "IOException ", e);
                }
            }

            File file = context.getFilesDir();
            if (file.exists()) {
                File oldData = new File(file, "vungle");
                try {
                    FileUtility.delete(oldData);
                } catch (IOException e) {
                    Log.e(TAG, "IOException ", e);
                }
            }

            //Delete cache dir downloads_vungle
            try {
                FileUtility.delete(new File(context.getCacheDir() + File.separator + "downloads_vungle"));
            } catch (IOException e) {
                Log.e(TAG, "IOException ", e);
            }
        }
    }

    private void runAndWait(Callable<Void> callable) throws DatabaseHelper.DBException {
        try {
            ioExecutor.submit(callable).get();
        } catch (ExecutionException e) {
            if (e.getCause() instanceof DatabaseHelper.DBException) {
                throw (DatabaseHelper.DBException) e.getCause();
            }
            Log.e(TAG, "Exception during runAndWait", e);
        } catch (InterruptedException e) {
            Log.e(TAG, "InterruptedException ", e);
            Thread.currentThread().interrupt();
        }
    }

    public <T> void deleteAll(Class<T> clazz){
        if (clazz == Advertisement.class) {
            List<Advertisement> advertisements = loadAll(Advertisement.class).get();
            for (Advertisement advertisement: advertisements) {
                try {
                    deleteAdvertisement(advertisement.getId());
                } catch (DatabaseHelper.DBException e) {
                    Log.e(TAG, "DB Exception deleting advertisement", e);
                }
            }
        } else {
            List<T> items = loadAll(clazz).get();
            for(T item: items) {
                try {
                    deleteModel(item);
                } catch (DatabaseHelper.DBException e) {
                    Log.e(TAG, "DB Exception deleting db entry", e);
                }
            }
        }

    }

    @VisibleForTesting
    public void setMockDBHelper(DatabaseHelper helper) {
        this.dbHelper = helper;
    }
}