package com.vungle.warren.tasks;

import android.os.Bundle;
import androidx.annotation.IntDef;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.vungle.warren.analytics.AdAnalytics;

public class AnalyticsJob implements Job {

    public static final String TAG = AnalyticsJob.class.getSimpleName();

    public static final String EXTRA_ACTION = "action_extra";
    private static final String EXTRA_BODY = "extra_body";
    private static final String EXTRA_URLS = "extra_urls";
    private static final long DEFAULT_DELAY = 2L * 1000L;

    private final AdAnalytics adAnalytics;

    public static JobInfo makeJob(@Action int action, String body, String[] urls, @JobInfo.NetworkType int networkType) {
        Bundle extras = new Bundle();
        extras.putInt(EXTRA_ACTION, action);
        extras.putString(EXTRA_BODY, body);
        extras.putStringArray(EXTRA_URLS, urls);

        return new JobInfo(TAG)
                .setUpdateCurrent(false)
                .setExtras(extras)
                .setReschedulePolicy(DEFAULT_DELAY, JobInfo.ReschedulePolicy.EXPONENTIAL)
                .setRequiredNetworkType(networkType)
                .setPriority(JobInfo.Priority.CRITICAL);
    }

    public AnalyticsJob(AdAnalytics adAnalytics) {
        this.adAnalytics = adAnalytics;
    }

    @Result
    @Override
    public int onRunJob(Bundle bundle, JobRunner jobRunner) {
        int action = bundle.getInt(EXTRA_ACTION, -1);
        if (action == Action.RI) {      //could be reworked to pass string
            String body = bundle.getString(EXTRA_BODY);
            Gson gson = new Gson();
            JsonElement element = gson.fromJson(body, JsonElement.class);
            JsonObject jsonObj = element.getAsJsonObject();
            adAnalytics.ri(jsonObj);
        } else if (action == Action.PING) {
            String[] urls = bundle.getStringArray(EXTRA_URLS);
            if (urls != null) {
                String[] failed = adAnalytics.ping(urls);
                if (failed.length != 0) {
                    bundle.putStringArray(EXTRA_URLS, failed);
                    return Result.RESCHEDULE;
                }
            }
        } else if (action == Action.RETRY_UNSENT) {
            String[] failed = adAnalytics.retryUnsent();
            if (failed.length != 0) {
                bundle.putStringArray(EXTRA_URLS, failed);
                return Result.RESCHEDULE;
            }
        } else if (action == Action.STORE_URL) {
            String[] urls = bundle.getStringArray(EXTRA_URLS);
            if (urls != null) {
                adAnalytics.saveVungleUrls(urls);
            }
        }

        return Result.SUCCESS;
    }

    @IntDef(value = {Action.RI, Action.PING, Action.RETRY_UNSENT, Action.STORE_URL})
    public @interface Action {
        int RI = 0;
        int PING = 1;
        int RETRY_UNSENT = 2;
        int STORE_URL = 3;
    }
}
