package com.vungle.warren.tasks;

import android.os.Bundle;
import android.os.SystemClock;
import android.util.Log;

import androidx.annotation.NonNull;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.vungle.warren.AdConfig;
import com.vungle.warren.AdLoader;
import com.vungle.warren.AdRequest;
import com.vungle.warren.CacheBustManager;
import com.vungle.warren.VungleApiClient;
import com.vungle.warren.VungleLogger;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.CacheBust;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.model.Placement;
import com.vungle.warren.network.Response;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Repository;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ExecutorService;

/**
 * Goal of cache bust job is to invalidate advertisements to be busted
 */
public class CacheBustJob implements Job {

    public static final String TAG = CacheBustJob.class.getCanonicalName();
    private static final String[] EMPTY_STRING_ARRAY = new String[0];

    private final VungleApiClient client;
    private final Repository repository;
    private final AdLoader adLoader;

    public CacheBustJob(@NonNull VungleApiClient client, @NonNull Repository repository, AdLoader adLoader) {
        this.client = client;
        this.repository = repository;
        this.adLoader = adLoader;
    }

    @Override
    public int onRunJob(Bundle bundle, JobRunner jobRunner) {

        Log.i(TAG, "CacheBustJob started");
        if (client == null || repository == null) {
            Log.e(TAG, "CacheBustJob finished - no client or repository");
            return Result.FAILURE;
        }

        try {
            Cookie cacheBustCookie = repository.load(Cookie.CACHE_BUST_COOKIE, Cookie.class).get();
            if (cacheBustCookie == null) {
                cacheBustCookie = new Cookie(Cookie.CACHE_BUST_COOKIE);
            }
            long lastCacheBustTime = cacheBustCookie.getLong(Cookie.LAST_CACHE_BUST);

            Response<JsonObject> response = client.cacheBust(lastCacheBustTime).execute();
            List<CacheBust> cacheBustList = new ArrayList<>();
            List<CacheBust> unProcessedBusts = repository.getUnProcessedBusts();
            if (unProcessedBusts != null && !unProcessedBusts.isEmpty()) {
                cacheBustList.addAll(unProcessedBusts);
            }

            Gson gson = new Gson();

            if (response.isSuccessful()) {
                JsonObject jsonObject = response.body();
                if (jsonObject == null || !jsonObject.has("cache_bust")) {
                    Log.e(TAG, "CacheBustJob finished - no jsonObject or cache_bust in it");
                    return Result.FAILURE;
                }

                JsonObject cacheBustObject = jsonObject.getAsJsonObject("cache_bust");
                if (cacheBustObject.has("last_updated")) {
                    long lastCacheBust = cacheBustObject.get("last_updated").getAsLong();
                    if (lastCacheBust > 0) {
                        cacheBustCookie.putValue(Cookie.LAST_CACHE_BUST, cacheBustObject.get("last_updated").getAsLong());
                        repository.save(cacheBustCookie);
                    }
                }
                parseAndSaveCacheBust(cacheBustObject, "campaign_ids", CacheBust.EVENT_TYPE_CAMPAIGN,
                        "cannot save campaignBust=", cacheBustList, gson);
                parseAndSaveCacheBust(cacheBustObject, "creative_ids", CacheBust.EVENT_TYPE_CREATIVE,
                        "cannot save creativeBust=", cacheBustList, gson);
            }
            processBust(cacheBustList);
            updateTimerData(bundle, cacheBustCookie);
        } catch (IOException e) {
            Log.e(TAG, "CacheBustJob failed - IOException", e);
            return Result.RESCHEDULE;
        } catch (DatabaseHelper.DBException e) {
            Log.e(TAG, "CacheBustJob failed - DBException", e);
            return Result.RESCHEDULE;
        }

        sendAnalytics();

        Log.d(TAG, "CacheBustJob finished");
        return Result.RESCHEDULE;
    }

    private void parseAndSaveCacheBust(JsonObject cacheBustObject, String fieldName,
                                       int cacheBustIdType, final String errorLogMessage,
                                       List<CacheBust> cacheBustList, Gson gson) {
        if (cacheBustObject.has(fieldName)) {
            for (JsonElement jsonElement : cacheBustObject.getAsJsonArray(fieldName)) {
                final CacheBust cacheBust = gson.fromJson(jsonElement, CacheBust.class);
                cacheBust.setTimeWindowEnd(cacheBust.getTimeWindowEnd() * 1000); /* server
                        would send timestamps in seconds, but under the hood we can stay adequate */
                cacheBust.setIdType(cacheBustIdType);
                cacheBustList.add(cacheBust);
                try {
                    repository.save(cacheBust);
                } catch (DatabaseHelper.DBException e) {
                    VungleLogger.error(CacheBustJob.class.getSimpleName() +
                            "#onRunJob", errorLogMessage + cacheBust);
                }
            }
        }
    }

    private void processBust(Iterable<CacheBust> busts) {
        for (CacheBust bust : busts) {
            List<Advertisement> adsInRepo = bust.getIdType() == CacheBust.EVENT_TYPE_CAMPAIGN
                    ? repository.getAdsByCampaign(bust.getId()) : repository.getAdsByCreative(bust.getId());
            List<String> eventIds = new LinkedList<>();
            List<Advertisement> adsToBust = new LinkedList<>();
            for (Advertisement adInRepo : adsInRepo) {
                if (adInRepo.getServerRequestTimestamp() < bust.getTimeWindowEnd() && shouldAdBeBusted(adInRepo)) {
                    eventIds.add(adInRepo.getId());
                    adsToBust.add(adInRepo);
                }
            }

            if (eventIds.isEmpty()) {
                Log.d(TAG, "processBust: bust has no relevant ads, deleting " + bust);
                try {
                    repository.delete(bust);
                } catch (DatabaseHelper.DBException e) {
                    VungleLogger.error(CacheBustJob.class.getSimpleName() + "#processBust",
                            "Cannot delete obsolete bust " + bust + " because of " + e);
                }
                continue;
            }

            bust.setEventIds(eventIds.toArray(EMPTY_STRING_ARRAY));
            for (Advertisement adToBust : adsToBust) {
                bustAd(adToBust, bust);
            }
        }
    }

    private boolean shouldAdBeBusted(Advertisement advertisement) {
        return advertisement.getState() != Advertisement.VIEWING &&
                advertisement.getState() != Advertisement.DONE;
    }

    private void sendAnalytics() {
        List<CacheBust> cacheBusts = repository.loadAll(CacheBust.class).get();
        if (cacheBusts == null || cacheBusts.size() == 0) {
            Log.d(TAG, "sendAnalytics: no cachebusts in repository");
            return;
        }
        //TODO implement this selection/filtering via SQLite in Repository
        Collection<CacheBust> bustsToSend = new LinkedList<>();
        for (CacheBust cacheBust : cacheBusts) {
            if (cacheBust.getTimestampProcessed() != 0) {
                bustsToSend.add(cacheBust);
            }
        }
        if (bustsToSend.isEmpty()) {
            Log.d(TAG, "sendAnalytics: no cachebusts to send analytics");
            return;
        }
        try {
            Response<JsonObject> bustAnalytics = client.sendAnalytics(bustsToSend).execute();
            if (bustAnalytics.isSuccessful()) {
                for (CacheBust cacheBust : bustsToSend) {
                    try {
                        repository.delete(cacheBust);
                    } catch (DatabaseHelper.DBException e) {
                        VungleLogger.error(CacheBustManager.class.getSimpleName() + "#sendAnalytics", "can't delete bust \" + cacheBust");
                    }
                }
            } else {
                Log.e(TAG, "sendAnalytics: not successful, aborting, response is " + bustAnalytics);
            }
        } catch (IOException e) {
            Log.e(TAG, "sendAnalytics: can't execute API call", e);
        }

    }

    private void bustAd(Advertisement advertisement, CacheBust cacheBust) {
        try {
            Log.d(TAG, "bustAd: deleting " + advertisement.getId());
            adLoader.dropCache(advertisement.getId());
            repository.deleteAdvertisement(advertisement.getId());
            Placement busted = repository.load(repository.getPlacementIdByAd(advertisement),
                    Placement.class).get();
            if (busted != null) {
                AdConfig config = new AdConfig();
                config.setAdSize(busted.getAdSize());
                if (busted.isMultipleHBPEnabled()) {
                    adLoader.loadEndlessIfNeeded(busted, busted.getAdSize(), 0, false);
                } else if (busted.isAutoCached()) {
                    adLoader.load(new AdLoader.Operation(
                            new AdRequest(busted.getId(), false),
                            busted.getAdSize(),
                            0,
                            AdLoader.RETRY_DELAY,
                            AdLoader.RETRY_COUNT,
                            AdLoader.ReschedulePolicy.EXPONENTIAL_ENDLESS_AD,
                            0,
                            false,
                            busted.getAutoCachePriority()
                    ));
                }
            }
            cacheBust.setTimestampProcessed(System.currentTimeMillis());
            repository.save(cacheBust);
        } catch (DatabaseHelper.DBException e) {
            Log.e(TAG, "bustAd: cannot drop cache or delete advertisement for " + advertisement, e);
        }
    }

    protected void updateTimerData(Bundle bundle, Cookie cacheBustCookie) throws DatabaseHelper.DBException {
        long cacheBustInterval = bundle.getLong(CacheBustManager.CACHE_BUST_INTERVAL);
        if (cacheBustInterval != 0L) {
            cacheBustCookie.putValue(CacheBustManager.NEXT_CACHE_BUST,
                    SystemClock.elapsedRealtime() + cacheBustInterval);
        }
        repository.save(cacheBustCookie);
    }

    public static JobInfo makeJobInfo() {
        return new JobInfo(TAG)
                .setPriority(JobInfo.Priority.LOWEST)
                .setUpdateCurrent(true);
    }
}