package com.vungle.warren.tasks;

import android.os.Bundle;
import android.util.Log;

import androidx.annotation.NonNull;

import com.vungle.warren.AdLoader;
import com.vungle.warren.session.SessionAttribute;
import com.vungle.warren.session.SessionEvent;
import com.vungle.warren.SessionTracker;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.Placement;
import com.vungle.warren.model.SessionData;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Designer;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.utility.FileUtility;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

/**
 * Module that scans through the asset and metadata caches for any expired or stale data, and removes
 * it. It also makes sure to redownload assets for any advertisement which is auto-cached.
 */
public class CleanupJob implements Job {

    static final String TAG = CleanupJob.class.getCanonicalName();

    private final Designer designer;
    private final Repository repository;
    private final AdLoader adLoader;

    /**
     * CleanupJob that will use the specified {@link Designer} and {@link Repository} to remove any
     * stale or expired data.
     *
     * @param designer The Designer instance to clean
     */
    CleanupJob(@NonNull Designer designer, @NonNull Repository repository, @NonNull AdLoader adLoader) {
        this.designer = designer;
        this.repository = repository;
        this.adLoader = adLoader;
    }

    @Override
    public int onRunJob(Bundle bundle, JobRunner jobRunner) {
        /*
         * Cleans up the metadata first, by scanning through the files in the files directory and
         * verifying that (1) the placement is still a valid placement, and (2) the advertisement
         * has not expired yet. We begin by making sure that invalid placements, their advertisements,
         * and the advertisement assets have been deleted.
         */
        if (designer == null || repository == null) {
            return Result.FAILURE;
        }

        /// Debugging information
        Log.d(TAG, "CleanupJob: Current directory snapshot");
        FileUtility.printDirectoryTree(designer.getCacheDirectory());

        File[] assets = designer.getCacheDirectory().listFiles();

        /// If we don't have any metadata yet, there is nothing to clean up, so end here.
        List<Placement> cachedPlacements = repository.loadAll(Placement.class).get();
        if (cachedPlacements == null || cachedPlacements.size() == 0) return Result.SUCCESS;

        /// Extract the placement data and verify it against the list of valid placements.
        Collection<Placement> validPlacements = repository.loadValidPlacements().get();

        Set<String> validIds = new HashSet<>();

        try {
            for (final Placement p : cachedPlacements) {
                //compare id?
                if (validPlacements != null && !validPlacements.isEmpty() && !validPlacements.contains(p)) {
                    Log.d(TAG, String.format(Locale.ENGLISH, "Placement %s is no longer valid, deleting it and its advertisement", p.getId()));
                    /// This placement is no longer valid, we need to delete the files and clean up any associated advertisements.
                    repository.delete(p);
                } else {
                    /// This is a valid placement, check to make sure the advertisement has not expired.
                    List<String> ads = repository.findAdsForPlacement(p.getId()).get();
                    if (ads != null) {
                        for (String advertisementId : ads) {
                            Advertisement advertisement = repository.load(advertisementId, Advertisement.class).get();
                            if (advertisement != null) {
                                if (advertisement.getExpireTime() <= System.currentTimeMillis()
                                        && advertisement.getState() != Advertisement.VIEWING) {
                                    /// This advertisement has expired. Delete the assets and break the link to this placement.
                                    repository.deleteAdvertisement(advertisementId);
                                    SessionTracker.getInstance().trackEvent(new SessionData.Builder().setEvent(SessionEvent.AD_EXPIRED)
                                            .addData(SessionAttribute.EVENT_ID, advertisementId).build());

                                    // If this is an auto-cached placement or HBP is enabled, schedule a download job for it
                                    adLoader.loadEndlessIfNeeded(p, p.getAdSize(), 1000, false);
                                } else {
                                    /// This advertisement is still valid. Save the advertisement ID to the list of valid
                                    /// advertisements. This list is used later in order to remove any leftover assets
                                    validIds.add(advertisement.getId());
                                    Log.w(TAG, "setting valid adv " + advertisementId + " for placement " + p.getId());
                                }
                            }
                        }
                    }
                }
            }

            //Add adv that are currently playing
            List<Advertisement> cachedAds = repository.loadAll(Advertisement.class).get();
            if (cachedAds != null) {
                for (Advertisement ad : cachedAds) {
                    if (ad.getState() == Advertisement.VIEWING) {
                        validIds.add(ad.getId());
                        Log.d(TAG, "found adv in viewing state " + ad.getId());
                    } else if (!validIds.contains(ad.getId())) {
                        Log.e(TAG, "    delete ad " + ad.getId());
                        repository.deleteAdvertisement(ad.getId());
                    }
                }
            }

            /// Remove assets in the asset cache directory if they are not part of a currently-valid advertisement.
            if (assets != null) {
                for (File f : assets) {
                    String id = f.getName();
                    if (!validIds.contains(id)) {
                        Log.v(TAG, String.format(Locale.ENGLISH, "Deleting assets under directory %s", f.getName()));
                        FileUtility.delete(f);
                    }
                }
            }

        } catch (DatabaseHelper.DBException ignored) {
            return Result.FAILURE;
        } catch (IOException e) {
            Log.e(TAG, "Failed to delete asset directory!", e);
            return Result.FAILURE;
        }
        return Result.SUCCESS;
    }

    public static JobInfo makeJobInfo() {
        return new JobInfo(TAG)
                .setPriority(JobInfo.Priority.LOWEST)
                .setUpdateCurrent(true);
    }
}
