package com.vungle.warren.tasks;

import android.os.Bundle;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.vungle.warren.AdLoader;
import com.vungle.warren.AdRequest;
import com.vungle.warren.VungleStaticApi;

import java.util.Collection;

/**
 * Download Job is responsible for downloading the data from a given URL to the given file. It will
 * schedule this work for when there is network connectivity available.
 */

public class DownloadJob implements Job {

    static final String TAG = DownloadJob.class.getCanonicalName();
    private static final String REQUEST_KEY_EXTRA = "request";
    private final AdLoader adLoader;
    private final VungleStaticApi vungleApi;

    public DownloadJob(@NonNull AdLoader adLoader, @NonNull VungleStaticApi vungleApi) {
        this.adLoader = adLoader;
        this.vungleApi = vungleApi;
    }

    public static JobInfo makeJobInfo(@NonNull AdRequest request) {
        Bundle extras = new Bundle();
        extras.putSerializable(REQUEST_KEY_EXTRA, request);

        String tag = TAG + " " + request;

        return new JobInfo(tag)
                .setUpdateCurrent(true)
                .setExtras(extras)
                .setPriority(JobInfo.Priority.HIGHEST);
    }

    @Override
    @Result
    public int onRunJob(Bundle params, JobRunner jobRunner) {
        AdRequest request = (AdRequest) params.getSerializable(REQUEST_KEY_EXTRA);

        /*
         * It is possible that a job was scheduled to download assets for a placement that is no
         * longer valid. Make sure that we're not attempting to download assets for an invalid
         * placement here.
         */
        Collection<String> validPlacements = vungleApi.getValidPlacements();

        if (request == null || !validPlacements.contains(request.getPlacementId())) {
            return Result.FAILURE;
        }

        adLoader.loadPendingInternal(request);

        return Result.SUCCESS;
    }
}
