package com.vungle.warren.tasks;

import android.os.Bundle;
import android.util.Log;

import com.vungle.warren.VungleApiClient;
import com.vungle.warren.model.Report;
import com.vungle.warren.network.Response;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Repository;

import java.io.IOException;
import java.util.List;


/**
 * Module that manages uploading advertisement reports. This Job is scheduled when the SDK is initialized
 * and after any advertisement is played. In the case of failure, another attempt will be scheduled. This
 * job will not run if there is no network connectivity, as it would immediately fail.
 */
public class SendReportsJob implements Job {

    private static final long DEFAULT_DELAY = 30000;
    static final String TAG = SendReportsJob.class.getCanonicalName();

    private Repository repository;
    private VungleApiClient vungleApiClient;

    public SendReportsJob(Repository repository, VungleApiClient vungleApiClient) {
        this.repository = repository;
        this.vungleApiClient = vungleApiClient;
    }

    public static JobInfo makeJobInfo(boolean sendAllReports) {
        Bundle extras = new Bundle();
        extras.putBoolean("sendAll", sendAllReports);

        return new JobInfo(TAG)
                .setExtras(extras)
                .setPriority(JobInfo.Priority.CRITICAL)
                .setReschedulePolicy(DEFAULT_DELAY, JobInfo.ReschedulePolicy.EXPONENTIAL);
    }

    @Override
    public int onRunJob(Bundle params, JobRunner jobRunner) {
        /// Debugging information
        boolean sendAllReports = params.getBoolean("sendAll", false);

        List<Report> reportList;

        if (sendAllReports) {
            reportList = repository.loadAllReportToSend().get(); // we are getting everything as we do not have enough qualifiers to filter
        } else {
            reportList = repository.loadReadyOrFailedReportToSend().get();
        }

        if (reportList == null) return Result.FAILURE;

        /// Otherwise, rehydrate these and send them off to the server
        for (final Report r : reportList) {
            try {
                Response response = vungleApiClient.reportAd(r.toReportBody()).execute();
                if (response.code() == 200) {
                    repository.delete(r);
                } else {
                    r.setStatus(Report.FAILED);
                    repository.save(r);
                    long retryAfterHeaderValue = vungleApiClient.getRetryAfterHeaderValue(response);
                    if (retryAfterHeaderValue > 0) {
                        jobRunner.execute(SendReportsJob.makeJobInfo(false).setDelay(retryAfterHeaderValue));
                        return Result.FAILURE;
                    }
                }

            } catch (IOException e) {
                Log.d(TAG, "SendReportsJob: IOEx");
                for (final Report markFail : reportList) {
                    markFail.setStatus(Report.FAILED);
                    try {
                        repository.save(markFail);
                    } catch (DatabaseHelper.DBException ignored) {
                        return Result.FAILURE;
                    }
                }
                Log.e(TAG, Log.getStackTraceString(e));
                return Result.RESCHEDULE;
            } catch (DatabaseHelper.DBException e) {
                return Result.FAILURE;
            }
        }

        return Result.SUCCESS;
    }

}
