package com.vungle.warren.tasks.runnable;

import android.os.Bundle;
import android.os.Process;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.util.Log;

import com.vungle.warren.tasks.Job;
import com.vungle.warren.tasks.JobCreator;
import com.vungle.warren.tasks.JobInfo;
import com.vungle.warren.tasks.JobRunner;
import com.vungle.warren.tasks.UnknownTagException;
import com.vungle.warren.tasks.utility.ThreadPriorityHelper;
import com.vungle.warren.utility.PriorityRunnable;


public class JobRunnable extends PriorityRunnable {

    private static final String TAG = JobRunnable.class.getSimpleName();

    private final JobInfo jobinfo;
    private final JobCreator creator;
    private final JobRunner jobRunner;
    private final ThreadPriorityHelper threadPriorityHelper;

    public JobRunnable(@NonNull JobInfo jobInfo, @NonNull JobCreator jobCreator, @NonNull JobRunner jobRunner, @Nullable ThreadPriorityHelper threadPriorityHelper) {
        this.jobinfo = jobInfo;
        this.creator = jobCreator;
        this.jobRunner = jobRunner;
        this.threadPriorityHelper = threadPriorityHelper;
    }

    @Override
    public @JobInfo.Priority Integer getPriority() {
        return jobinfo.getPriority();
    }

    @Override
    public void run() {

        if (threadPriorityHelper != null) {
            try {
                int targetPriority = threadPriorityHelper.makeAndroidThreadPriority(jobinfo);
                Process.setThreadPriority(targetPriority);
                Log.d(TAG, "Setting process thread prio = " + targetPriority + " for " + jobinfo.getJobTag());
            } catch (Throwable throwable) {
                Log.e(TAG, "Error on setting process thread priority");
            }
        }

        try {

            String jobTag = jobinfo.getJobTag();
            Bundle params = jobinfo.getExtras();

            Log.d(TAG, "Start job " + jobTag + "Thread " + Thread.currentThread().getName());
            Job job = creator.create(jobTag);

            int result = job.onRunJob(params, jobRunner);
            Log.d(TAG, "On job finished " + jobTag + " with result " + result);

            if (result == Job.Result.RESCHEDULE) {
                long nextReschedule = jobinfo.makeNextRescedule();

                if (nextReschedule > 0) {
                    jobinfo.setDelay(nextReschedule);
                    jobRunner.execute(jobinfo);
                    Log.d(TAG, "Rescheduling " + jobTag + " in " + nextReschedule);
                }

            }

        } catch (UnknownTagException e) {
            Log.e(TAG, "Cannot create job" + e.getLocalizedMessage());
        } catch (Throwable th) {
            Log.e(TAG, "Can't start job", th);
        }
    }
}
