package com.vungle.warren.ui.presenter;

import android.app.AlertDialog;
import android.content.ActivityNotFoundException;
import android.content.DialogInterface;
import android.content.pm.ActivityInfo;
import android.text.TextUtils;
import android.util.Log;

import com.vungle.warren.AdConfig;
import com.vungle.warren.VungleLogger;
import com.vungle.warren.analytics.AdAnalytics;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.model.Placement;
import com.vungle.warren.model.Report;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.ui.DurationRecorder;
import com.vungle.warren.ui.PresenterAdOpenCallback;
import com.vungle.warren.ui.PresenterAppLeftCallback;
import com.vungle.warren.ui.contract.AdContract;
import com.vungle.warren.ui.contract.NativeAdContract;
import com.vungle.warren.ui.state.OptionsState;
import com.vungle.warren.utility.Constants;
import com.vungle.warren.utility.Scheduler;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import static com.vungle.warren.analytics.AnalyticsEvent.Ad.videoLength;
import static com.vungle.warren.error.VungleException.DB_ERROR;
import static com.vungle.warren.model.Cookie.CONFIG_COOKIE;
import static com.vungle.warren.model.Cookie.CONSENT_COOKIE;
import static com.vungle.warren.model.Cookie.INCENTIVIZED_TEXT_COOKIE;
import static com.vungle.warren.ui.contract.AdContract.AdStopReason.IS_CHANGING_CONFIGURATION;

/**
 * Presenter that handles the business logic of showing a native ad. Given an {@link Advertisement}
 * and {@link Placement}, this presenter manages the {@link AdContract.AdView} which is showing the ad. The
 * presenter is also responsible for creating the ad report and managing any extra components, such
 * as the event listener, which are not the responsibility of the view layer.
 */
public class NativeAdPresenter implements NativeAdContract.NativePresenter {

    private static final String TAG = NativeAdPresenter.class.getSimpleName();
    private static final String EXTRA_REPORT = "saved_report";
    private static final String EXTRA_INCENTIVIZED_SENT = "incentivized_sent";
    private static final String OPEN_ACTION = "mraidOpen";
    private static final String OPEN_DEEPLINK_SUCCESS = "deeplinkSuccess";

    private static final String NO_VALUE = "";

    /**
     * The advertisement metadata. Includes information about where to locate the assets, as well
     * as internal communicaiton information, such as the third-party attribution URLs.
     */
    private final Advertisement advertisement;

    /**
     * The placement being presenter to the user. This reference holds information about business
     * logic, such as if this advertisement is incentivized.
     */
    private final Placement placement;

    /**
     * Reference to the SDK's persistor implementation, which manages serializing and deserializing
     * data from disk.
     */
    private final Repository repository;
    private final Scheduler scheduler;
    private final AdAnalytics analytics;

    private final String[] impressionUrls;

    /**
     * The advertisement report which we will send to the AdServer once the advertisement has finished.
     * It includes all of the information about how the advertisement was played, and for how long.
     */
    private Report report;

    private final Map<String, Cookie> cookies = new HashMap<>();

    /**
     * The view which is displaying the advertisement and taking direction from this presenter.
     */
    private NativeAdContract.NativeView adView;

    /**
     * track if adviewed callback has been invoked. cannot send more than once per ad
     */
    private boolean adViewed;

    /**
     * The event bus where we emit lifecycle information, such as "start", "end", and "error".
     */
    private AdContract.AdvertisementPresenter.EventListener bus;

    private final AtomicBoolean sendReportIncentivized = new AtomicBoolean(false);
    private final AtomicBoolean isDestroying = new AtomicBoolean(false);
    private final LinkedList<Advertisement.Checkpoint> checkpointList = new LinkedList<>();
    private final Repository.SaveCallback repoCallback = new Repository.SaveCallback() {
        boolean errorHappened = false;

        @Override
        public void onSaved() {

        }

        @Override
        public void onError(Exception e) {
            if (errorHappened)
                return;
            errorHappened = true;
            makeBusError(DB_ERROR);
            VungleLogger.error(
                    LocalAdPresenter.class.getSimpleName() + "#onError",
                    new VungleException(VungleException.DB_ERROR).getLocalizedMessage());
            closeView();
        }
    };

    private DurationRecorder durationRecorder;

    public NativeAdPresenter(@NonNull Advertisement advertisement,
                             @NonNull Placement placement,
                             @NonNull Repository repository,
                             @NonNull Scheduler scheduler,
                             @NonNull AdAnalytics analytics,
                             @Nullable OptionsState state,
                             @Nullable String[] impressionUrls) {
        this.advertisement = advertisement;
        this.placement = placement;
        this.repository = repository;
        this.scheduler = scheduler;
        this.analytics = analytics;
        this.impressionUrls = impressionUrls;

        if (advertisement.getCheckpoints() != null) {
            checkpointList.addAll(advertisement.getCheckpoints());
        }

        loadData(state);
    }

    @Override
    public void onDownload() {
        reportAction(OPEN_ACTION, NO_VALUE);

        try {
            analytics.ping(advertisement.getTpatUrls("clickUrl"));
            analytics.ping(new String[]{advertisement.getCTAURL(true)});

            reportAction(DOWNLOAD_ACTION, null);
            String url = advertisement.getCTAURL(false);
            String deeplinkUrl = advertisement.getDeeplinkUrl();
            if ((deeplinkUrl == null || deeplinkUrl.isEmpty()) && (url == null || url.isEmpty())) {
                Log.e(TAG, "CTA destination URL is not configured properly");
            } else {
                adView.open(deeplinkUrl, url, new PresenterAppLeftCallback(bus, placement), new PresenterAdOpenCallback() {
                    @Override
                    public void onAdOpenType(AdOpenType adOpenType) {
                        if (adOpenType == AdOpenType.DEEP_LINK) {
                            reportAction(OPEN_DEEPLINK_SUCCESS, null);
                        }
                    }
                });
            }

            if (bus != null) {
                bus.onNext("open", "adClick", placement.getId());
            }
        } catch (ActivityNotFoundException invalid) {
            Log.e(TAG, "Unable to find destination activity");
            VungleLogger.error(
                    LocalAdPresenter.class.getSimpleName() + "#download",
                    "Download - Activity Not Found");
        }
    }

    @Override
    public void onPrivacy() {
        String privacyUrl = advertisement.getPrivacyUrl();
        adView.open(null, privacyUrl, new PresenterAppLeftCallback(bus, placement), null);
    }

    @Override
    public void setAdVisibility(boolean isViewable) {
        Log.d(TAG, "isViewable=" + isViewable + " " + placement + " " + hashCode());
        if (isViewable) {
            durationRecorder.start();
        } else {
            durationRecorder.stop();
        }
    }

    @Override
    public void attach(@NonNull NativeAdContract.NativeView adView, @Nullable OptionsState state) {
        Log.d(TAG, "attach() " + placement + " " + hashCode());
        isDestroying.set(false);
        this.adView = adView;
        adView.setPresenter(this);
        if (bus != null) {
            bus.onNext(Constants.ATTACH, advertisement.getCreativeId(), placement.getId());
        }

        // Check if the advertisement has a specified orientation, then lock the window to that
        int requestedOrientation = -1; /// Sentinel value
        int adOrientation = advertisement.getAdConfig().getAdOrientation();
        if (adOrientation == AdConfig.MATCH_VIDEO) {
            switch (advertisement.getOrientation()) {
                case Advertisement.PORTRAIT:
                    /// Lock the orientation to portrait.
                    requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT;
                    break;
                case Advertisement.LANDSCAPE:
                    /// Lock the orientation to landscape.
                    requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE;
                    break;
            }
        } else if (adOrientation == AdConfig.PORTRAIT) {
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT;
        } else if (adOrientation == AdConfig.LANDSCAPE) {
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE;
        } else {
            requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR;
        }

        Log.d(TAG, "Requested Orientation " + requestedOrientation);
        adView.setOrientation(requestedOrientation);
        prepare(state);
    }

    private void prepare(OptionsState state) {
        restoreFromSave(state);

        Cookie incentivizedCookie = cookies.get(INCENTIVIZED_TEXT_COOKIE);
        String userIdFromCookie = (incentivizedCookie == null) ? null : incentivizedCookie.getString("userID");

        if (report == null) {
            report = new Report(advertisement, placement, System.currentTimeMillis(), userIdFromCookie);
            report.setTtDownload(advertisement.getTtDownload());
            repository.save(report, repoCallback);
        }

        if (durationRecorder == null) {
            durationRecorder = new DurationRecorder(report, repository, repoCallback);
        }

        if (bus != null) {
            bus.onNext("start", null, placement.getId());
        }
    }

    @Override
    public void detach(@AdContract.AdStopReason int stopReason) {
        Log.d(TAG, "detach() " + placement + " " + hashCode());
        stop(stopReason);
        adView.destroyAdView(0L);
    }

    @Override
    public boolean handleExit() {
        closeView();
        return true;
    }

    @Override
    public void start() {
        Log.d(TAG, "start() " + placement + " " + hashCode());
        durationRecorder.start();

        /// Check for GDPR Consent State
        Cookie gdprConsent = cookies.get(CONSENT_COOKIE);
        if (needShowGDPR(gdprConsent)) {
            showGDPR(gdprConsent);
        }
    }

    @Override
    public void stop(@AdContract.AdStopReason int stopReason) {
        Log.d(TAG, "stop() " + placement + " " + hashCode());
        durationRecorder.stop();

        boolean isChangingConfigurations = (stopReason & IS_CHANGING_CONFIGURATION) != 0;
        boolean isFinishing = (stopReason & AdContract.AdStopReason.IS_AD_FINISHING) != 0;
        boolean isFinishByAPI = (stopReason & AdContract.AdStopReason.IS_AD_FINISHED_BY_API) != 0;

        if (!isChangingConfigurations && isFinishing && !isDestroying.getAndSet(true)) {
            if (isFinishByAPI) {
                reportAction("mraidCloseByApi", null);
            }

            repository.save(report, repoCallback);

            closeView();

            if (bus != null) {
                bus.onNext("end", report.isCTAClicked() ? "isCTAClicked" : null, placement.getId());
            }
        }
    }

    @Override
    public void generateSaveState(@Nullable OptionsState state) {
        if (state == null) {
            return;
        }

        repository.save(report, repoCallback);
        state.put(EXTRA_REPORT, report == null ? null : report.getId());
        state.put(EXTRA_INCENTIVIZED_SENT, sendReportIncentivized.get());
    }

    @Override
    public void restoreFromSave(@Nullable OptionsState state) {
        if (state == null) {
            return;
        }

        boolean incentivizedShown = state.getBoolean(EXTRA_INCENTIVIZED_SENT, false);

        if (incentivizedShown) {
            sendReportIncentivized.set(incentivizedShown);
        }

        //old code
        if (report == null) {
            /// The advertisement was not started and cannot be restored.
            adView.close();
            VungleLogger.error(
                    MRAIDAdPresenter.class.getSimpleName() + "#restoreFromSave",
                    "The advertisement was not started and cannot be restored.");
            return;
        }
    }

    @Override
    public void setEventListener(@Nullable EventListener listener) {
        this.bus = listener;
    }

    @Override
    public void onViewConfigurationChanged() {
        adView.refreshDialogIfVisible();
    }

    @Override
    public void onMraidAction(String action) {
        //no-op
    }

    @Override
    public void onProgressUpdate(int position, float duration) {
        Log.d(TAG, "onProgressUpdate() " + placement + " " + hashCode());
        if (bus != null && position > 0 && !adViewed) {
            adViewed = true;
            bus.onNext("adViewed", null, placement.getId());
            if (impressionUrls != null) {
                analytics.ping(impressionUrls);
            }
        }

        if (bus != null) {
            bus.onNext("percentViewed:100", null, placement.getId());
        }

        reportVideoLength(5000);
        reportAction(videoLength, String.format(Locale.ENGLISH, "%d", 5000));

        reportAction("videoViewed", String.format(Locale.ENGLISH, "%d", 100));

        Advertisement.Checkpoint checkpointStart = checkpointList.pollFirst();
        if (checkpointStart != null) {
            analytics.ping(checkpointStart.getUrls());
        }

        durationRecorder.update();
    }

    private void reportVideoLength(long value) {
        report.setVideoLength(value);
        repository.save(report, repoCallback);
    }

    private void reportAction(@NonNull String action, @Nullable String value) {
        report.recordAction(action, value, System.currentTimeMillis());
        repository.save(report, repoCallback);
    }

    private void loadData(OptionsState optionsState) {
        cookies.put(INCENTIVIZED_TEXT_COOKIE, repository.load(INCENTIVIZED_TEXT_COOKIE, Cookie.class).get());
        cookies.put(CONSENT_COOKIE, repository.load(CONSENT_COOKIE, Cookie.class).get());
        cookies.put(CONFIG_COOKIE, repository.load(CONFIG_COOKIE, Cookie.class).get());

        if (optionsState != null) {
            String reportId = optionsState.getString(EXTRA_REPORT);
            Report restoredReport = TextUtils.isEmpty(reportId) ? null : repository.load(reportId, Report.class).get();

            if (restoredReport != null) {
                report = restoredReport;
            }
        }
    }

    private boolean needShowGDPR(@Nullable Cookie gdprConsent) {
        return gdprConsent != null && gdprConsent.getBoolean("is_country_data_protected")
                && "unknown".equals(gdprConsent.getString("consent_status"));
    }

    private void showGDPR(@NonNull Cookie gdprConsent) {
        final Cookie finalGdpr = gdprConsent;
        AlertDialog.OnClickListener listener = new AlertDialog.OnClickListener() {

            @Override
            public void onClick(DialogInterface dialogInterface, int i) {
                String consented = "opted_out_by_timeout";
                if (i == AlertDialog.BUTTON_NEGATIVE) {
                    consented = "opted_out";
                } else if (i == AlertDialog.BUTTON_POSITIVE) {
                    consented = "opted_in";
                }

                finalGdpr.putValue("consent_status", consented);
                finalGdpr.putValue("timestamp", System.currentTimeMillis() / 1000);
                finalGdpr.putValue("consent_source", "vungle_modal");
                repository.save(finalGdpr, null);

                start();
            }
        };

        //by default save the user decision as opted out until user makes a choice
        gdprConsent.putValue("consent_status", "opted_out_by_timeout");
        finalGdpr.putValue("timestamp", System.currentTimeMillis() / 1000);
        finalGdpr.putValue("consent_source", "vungle_modal");
        repository.save(finalGdpr, repoCallback);

        /// Strings will come from the server for language appropriate to user.
        showDialog(gdprConsent.getString("consent_title"),
                gdprConsent.getString("consent_message"),
                gdprConsent.getString("button_accept"),
                gdprConsent.getString("button_deny"),
                listener);
    }

    private void showDialog(String titleText, String bodyText, String continueText, String closeText, DialogInterface.OnClickListener listener) {
        adView.showDialog(titleText, bodyText, continueText, closeText, listener);
    }

    private void makeBusError(@VungleException.ExceptionCode int code) {
        if (bus != null) {
            bus.onError(new VungleException(code), placement.getId());
        }
    }

    private void closeView() {
        adView.close();
        scheduler.cancelAll();
    }
}
