package com.vungle.warren.ui.view;

import android.content.Context;
import android.content.DialogInterface;
import android.media.MediaPlayer;
import android.net.Uri;
import android.os.Handler;
import android.os.Looper;
import androidx.annotation.NonNull;
import android.util.Log;

import com.vungle.warren.ui.CloseDelegate;
import com.vungle.warren.ui.OrientationDelegate;
import com.vungle.warren.ui.contract.LocalAdContract;
import com.vungle.warren.ui.presenter.LocalAdPresenter;

import java.io.File;

import static com.vungle.warren.ui.view.FullAdWidget.ViewEvent.CLOSE_CLICK;
import static com.vungle.warren.ui.view.FullAdWidget.ViewEvent.CTA_CLICK;
import static com.vungle.warren.ui.view.FullAdWidget.ViewEvent.MUTE_CLICK;
import static com.vungle.warren.ui.view.FullAdWidget.ViewEvent.PRIVACY_CLICK;
import static com.vungle.warren.ui.view.FullAdWidget.ViewEvent.VIDEO_CLICK;

/**
 * Created by Alexander Semenov on 10-Aug-18.
 */

public class LocalAdView extends BaseAdView<LocalAdPresenter> implements LocalAdContract.LocalView, MediaPlayer.OnErrorListener, MediaPlayer.OnPreparedListener {

    private LocalAdContract.LocalPresenter presenter;

    private boolean muted = false;
    private MediaPlayer mediaPlayer;

    private boolean ctaClickAreaEnabled = false;

    /**
     * Loop that triggers updates to be given to the presenter.
     */
    private Runnable reportProgress;

    /**
     * Handler that manages the reporting runnables, checking on a set interval the latest progress
     * and state of the advertisement and appends it to the current Report.
     */
    private Handler handler = new Handler(Looper.getMainLooper());

    public LocalAdView(@NonNull Context context,
                       @NonNull FullAdWidget view,
                       @NonNull OrientationDelegate orientationDelegate,
                       @NonNull CloseDelegate closeDelegate) {
        super(context, view, orientationDelegate, closeDelegate);
        attachListeners();
    }

    //Wrap with abstraction
    private void attachListeners() {
        view.setOnItemClickListener(clickListener);
        view.setOnPreparedListener(this);
        view.setOnErrorListener(this);
    }

    @Override
    public void showCTAOverlay(boolean ctaOverlayEnabled, boolean ctaClickAreaEnabled) {
        this.ctaClickAreaEnabled = ctaClickAreaEnabled;

        // this is a specialized requirement , that if click_area is set to 0 or disabled, that we do not honor enabled flag
        // effectively click_area trumps the enabled flag
        view.setCtaEnabled(ctaOverlayEnabled && ctaClickAreaEnabled);
    }

    @Override
    public void playVideo(@NonNull File file, boolean startMuted, int videoPosition) {
        muted = muted || startMuted;

        if (file != null) {
            setupPlayerProgressBar();
            view.playVideo(Uri.fromFile(file), videoPosition);
            view.setMuted(muted);

            if (muted) {
                //TODO: how do we know if mute is done by developers and user action?
                presenter.onMute(muted);
            }

        }
    }

    @Override
    public void pauseVideo() {
        view.pausePlayback();
        if (reportProgress != null) {
            handler.removeCallbacks(reportProgress);
        }
    }

    @Override
    public boolean isVideoPlaying() {
        return view.isVideoPlaying();
    }

    @Override
    public int getVideoPosition() {
        return view.getCurrentVideoPosition();
    }

    @Override
    public void setPresenter(@NonNull LocalAdPresenter presenter) {
        this.presenter = presenter;
    }

    @Override
    public void showWebsite(@NonNull String url) {
        view.stopPlayback();
        view.showWebsite(url);
        //fixes a bug that can cause crashes since:
        //reportProgress runs every second, mediaplayer can be null before then
        handler.removeCallbacks(reportProgress);
        mediaPlayer = null;
    }


    private FullAdWidget.OnItemClickListener clickListener = new FullAdWidget.OnItemClickListener() {
        @Override
        public void onItemClicked(int code) {
            switch (code) {

                case PRIVACY_CLICK:
                    presenter.onPrivacy();
                    break;

                case CLOSE_CLICK:
                    presenter.handleExit();
                    break;

                case CTA_CLICK:
                    presenter.onDownload();
                    break;

                case VIDEO_CLICK:
                    if (ctaClickAreaEnabled) {
                        presenter.onDownload();
                    }
                    break;

                case MUTE_CLICK:
                    if (mediaPlayer != null) {
                        muteAudio();
                        presenter.onMute(muted);
                        view.setMuted(muted);
                    }
                    break;
            }
        }
    };

    private void muteAudio() {
        if (mediaPlayer == null)
            return;

        muted = !muted;
        switchMuteState();
    }

    private void switchMuteState() {
        if (mediaPlayer != null) {
            try {
                float volume = muted ? 0.0f : 1.0f;
                mediaPlayer.setVolume(volume, volume);
            } catch (IllegalStateException e) {
                Log.i(TAG, "Exception On Mute/Unmute", e);
            }
        }
    }

    //TODO: rework responsibilities
    private void setupPlayerProgressBar() {
        /// Set a handler to report the progress of the video as it plays
        reportProgress = new Runnable() {
            /**
             * Sentinel value of -2 if not set, we want to only retrieve the duration of the video once,
             * since it is a heavy operation through the JNI layer to retrieve it.
             */
            float duration = -2f;

            @Override
            public void run() {

                try {

                    if (view.isVideoPlaying()) {
                        int currentPosition = view.getCurrentVideoPosition();
                        int currentDuration = view.getVideoDuration();

                        if (currentDuration > 0) {
                            if (duration == -2f) {
                                duration = (float) currentDuration;
                            }

                            presenter.onProgressUpdate(currentPosition, duration);
                            view.setProgress(currentPosition, duration);
                        }

                    }

                    handler.postDelayed(this, 1_000); /// Check progress again in 1 second.
                } catch (IllegalStateException released) {
                    /// Error occurs if this runloop is triggered after the activity has finished. This is
                    /// a rare race condition that only occurs if the activity was killed via SIGKILL
                    /// while the application was not. This causes onStop() to not be called and
                    /// therefore our cleanup logic for this Runnable is not run. This error means that
                    /// we should no longer report progress.
                    Log.v(TAG, "IllegalStateException while reporting progress indicates activity was killed via SIGKILL.");
                }
            }
        };
        handler.post(reportProgress);
    }

    @Override
    public boolean onError(MediaPlayer mediaPlayer, int what, int extra) {
        StringBuilder description = new StringBuilder(30);

        switch (what) {
            case MediaPlayer.MEDIA_ERROR_SERVER_DIED: {
                description.append("MEDIA_ERROR_SERVER_DIED");
                break;
            }
            case MediaPlayer.MEDIA_ERROR_UNKNOWN: {
                description.append("MEDIA_ERROR_UNKNOWN");
                break;
            }
            default: {
                description.append("UNKNOWN");
            }
        }

        description.append(':');
        switch (extra) {
            case MediaPlayer.MEDIA_ERROR_IO: {
                description.append("MEDIA_ERROR_IO");
                break;
            }
            case MediaPlayer.MEDIA_ERROR_MALFORMED: {
                description.append("MEDIA_ERROR_MALFORMED");
                break;
            }
            case MediaPlayer.MEDIA_ERROR_NOT_VALID_FOR_PROGRESSIVE_PLAYBACK: {
                description.append("MEDIA_ERROR_NOT_VALID_FOR_PROGRESSIVE_PLAYBACK");
                break;
            }
            case MediaPlayer.MEDIA_ERROR_UNSUPPORTED: {
                description.append("MEDIA_ERROR_UNSUPPORTED");
                break;
            }
            case MediaPlayer.MEDIA_ERROR_TIMED_OUT: {
                description.append("MEDIA_ERROR_TIMED_OUT");
                break;
            }
            default: {
                description.append("MEDIA_ERROR_SYSTEM");
            }
        }

        /// Report the error to the presenter, which will decide on the course of action based on the
        /// error type.
        presenter.onMediaError(description.toString());
        return true;
    }

    @Override
    public void onPrepared(final MediaPlayer mp) {
        this.mediaPlayer = mp;

        switchMuteState();

        view.setOnCompletionListener(new MediaPlayer.OnCompletionListener() {
            @Override
            public void onCompletion(final MediaPlayer mp) {
                Log.d(TAG, "mediaplayer onCompletion");
                /// Record the action that the advertisement was viewed, cleanup reporting runnable
                if (reportProgress != null) {
                    handler.removeCallbacks(reportProgress);
                }

                presenter.onProgressUpdate(mp.getDuration(), mp.getDuration());
            }
        });
        /// Inform the presenter about the length of the advertisement
        presenter.onVideoStart(getVideoPosition(), mp.getDuration());

        setupPlayerProgressBar();
    }

    @Override
    public void close() {
        super.close();
        handler.removeCallbacksAndMessages(null);
    }

}
