package com.vungle.warren.ui.view;

import android.app.AlertDialog;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.text.TextUtils;
import android.util.Log;
import android.view.ContextThemeWrapper;

import com.vungle.warren.NativeAdLayout;
import com.vungle.warren.ui.PresenterAdOpenCallback;
import com.vungle.warren.ui.contract.NativeAdContract;
import com.vungle.warren.ui.presenter.NativeAdPresenter;
import com.vungle.warren.utility.ActivityManager;
import com.vungle.warren.utility.ExternalRouter;

import org.jetbrains.annotations.NotNull;

import java.util.concurrent.atomic.AtomicReference;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import static com.vungle.warren.NativeAdLayout.ViewEvent.CTA_CLICK;
import static com.vungle.warren.NativeAdLayout.ViewEvent.PRIVACY_CLICK;

public class NativeAdView implements NativeAdContract.NativeView, NativeAdLayout.OnItemClickListener {

    private static final String TAG = NativeAdView.class.getSimpleName();

    private final Context context;

    private final NativeAdLayout view;

    private NativeAdContract.NativePresenter presenter;

    private Dialog currentDialog;

    public NativeAdView(@NonNull Context context, @NonNull NativeAdLayout view) {
        this.context = context;
        this.view = view;

        view.setOnItemClickListener(this);
    }

    @Override
    public void setPresenter(@NonNull NativeAdPresenter presenter) {
        this.presenter = presenter;
    }

    @Override
    public void setOrientation(int orientation) {
        // no-op
    }

    @Override
    public void showWebsite(@NonNull String url) {
        // no-op
    }

    @Override
    public String getWebsiteUrl() {
        return null;
    }

    @Override
    public void close() {
        // no-op
    }

    @Override
    public void showCloseButton() {
        // no-op
    }

    @Override
    public void open(String deeplinkUrl, @NonNull String url, ActivityManager.LeftApplicationCallback leftApplicationCallback, PresenterAdOpenCallback adOpenCallback) {
        if (!ExternalRouter.launch(deeplinkUrl, url, context, leftApplicationCallback, true, adOpenCallback)) {
            Log.e(TAG, "Cannot open url " + url);
        }
    }

    @Override
    public void showDialog(@Nullable String dialogTitle,
                           @Nullable String dialogBody,
                           @NonNull String dialogContinue,
                           @NonNull String dialogClose,
                           @Nullable final DialogInterface.OnClickListener responseListener) {
        AlertDialog.Builder dialogBuilder = new AlertDialog.Builder(new ContextThemeWrapper(context, context.getApplicationInfo().theme));

        DialogInterface.OnDismissListener dismissListener = getDialogDismissListener();

        DialogInterface.OnClickListener onClickListenerProxy = new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                currentDialog = null;
                if (responseListener != null) {
                    responseListener.onClick(dialog, which);
                }
            }
        };

        final DialogClickListenerProxy listener = new DialogClickListenerProxy(onClickListenerProxy, dismissListener);

        if (!TextUtils.isEmpty(dialogTitle)) {
            dialogBuilder.setTitle(dialogTitle);
        }

        if (!TextUtils.isEmpty(dialogBody)) {
            dialogBuilder.setMessage(dialogBody);
        }

        dialogBuilder.setPositiveButton(dialogContinue, listener);
        dialogBuilder.setNegativeButton(dialogClose, listener);

        dialogBuilder.setCancelable(false);
        currentDialog = dialogBuilder.create();

        listener.autoRelease(currentDialog);

        currentDialog.show();
    }

    private static class DialogClickListenerProxy implements DialogInterface.OnClickListener, Dialog.OnDismissListener {

        private final AtomicReference<DialogInterface.OnClickListener> ckickRef = new AtomicReference<>();
        private final AtomicReference<DialogInterface.OnDismissListener> dismissRef = new AtomicReference<>();

        public DialogClickListenerProxy(DialogInterface.OnClickListener onClickListener, DialogInterface.OnDismissListener onDismissListener) {
            ckickRef.set(onClickListener);
            dismissRef.set(onDismissListener);
        }

        @Override
        public void onClick(DialogInterface dialog, int which) {
            dialog.dismiss();
            DialogInterface.OnClickListener base = ckickRef.get();
            if (base != null) {
                base.onClick(dialog, which);
            }
        }

        private void autoRelease(Dialog dialog) {
            dialog.setOnDismissListener(this);
        }

        @Override
        public void onDismiss(DialogInterface dialog) {
            DialogInterface.OnDismissListener base = dismissRef.get();
            if (base != null) {
                base.onDismiss(dialog);
            }
            dismissRef.set(null);
            ckickRef.set(null);
        }
    }

    @NonNull
    protected DialogInterface.OnDismissListener getDialogDismissListener() {
        return new DialogInterface.OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {
                currentDialog = null;
            }
        };
    }

    @Override
    public void setImmersiveMode() {
        // no-op
    }

    @Override
    public void resumeWeb() {
        // no-op
    }

    @Override
    public void pauseWeb() {
        // no-op
    }

    @Override
    public void removeWebView() {
        // no-op
    }

    @Override
    public boolean hasWebView() {
        return false;
    }

    @Override
    public void destroyAdView(long webViewDestroyDelay) {
        view.release();
    }

    @Override
    public void refreshDialogIfVisible() {
        if (isDialogVisible()) {
            //ignore subsequent dismiss
            currentDialog.setOnDismissListener(new DialogInterface.OnDismissListener() {
                @Override
                public void onDismiss(DialogInterface dialog) {
                    currentDialog.setOnDismissListener(getDialogDismissListener());
                }
            });
            currentDialog.dismiss();
            currentDialog.show();
        }
    }

    @Override
    public boolean isDialogVisible() {
        return currentDialog != null;
    }

    @Override
    public void onItemClicked(int code) {
        switch (code) {
            case PRIVACY_CLICK:
                presenter.onPrivacy();
                break;

            case CTA_CLICK:
                presenter.onDownload();
                break;

        }
    }
}
