package com.vungle.warren.ui.view;

import android.annotation.SuppressLint;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentFilter;
import android.graphics.Color;
import android.os.Build;
import android.util.Log;
import android.util.Pair;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewManager;
import android.view.ViewParent;
import android.webkit.WebView;

import com.vungle.warren.AdConfig;
import com.vungle.warren.AdRequest;
import com.vungle.warren.AdvertisementPresentationFactory;
import com.vungle.warren.PresentationFactory;
import com.vungle.warren.SessionTracker;
import com.vungle.warren.VungleLogger;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.SessionData;
import com.vungle.warren.session.SessionAttribute;
import com.vungle.warren.session.SessionEvent;
import com.vungle.warren.ui.CloseDelegate;
import com.vungle.warren.ui.JavascriptBridge;
import com.vungle.warren.ui.PresenterAdOpenCallback;
import com.vungle.warren.ui.contract.AdContract;
import com.vungle.warren.ui.contract.WebAdContract;
import com.vungle.warren.utility.ActivityManager;
import com.vungle.warren.utility.ExternalRouter;
import com.vungle.warren.utility.HandlerScheduler;

import java.util.concurrent.atomic.AtomicReference;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;

/**
 * Banner ad view is a view returned by Vungle.getBannerViewInternal(). It extends WebView
 * and fills the width and height of the parent container. This method will return
 * null when Vungle is not initialized.
 */
public class VungleBannerView extends WebView implements WebAdContract.WebAdView {

    private static final String TAG = VungleBannerView.class.getName();
    /**
     * TODO: Missing javadoc
     */
    private WebAdContract.WebAdPresenter presenter;
    private BroadcastReceiver broadcastReceiver;
    private final AdContract.AdvertisementPresenter.EventListener listener;
    private final AdRequest request;
    private final AdConfig config;

    /**
     * How long we wait for an ad to start playing before we end it and call error
     * Mostly likely from an invalid resource or invalid url. We'll have to be more particular in the future
     * and narrow down exactly why ads are not showing. If this is the case we should send error events
     * upstream to server
     * <p>
     * In testing time to start video currently ranges anywhere from 500ms to 2500ms on Pixel 2
     */

    PresentationFactory presenterFactory;
    private AtomicReference<Boolean> isAdVisible = new AtomicReference<>();
    private boolean destroyed;

    @Override
    public void onResume() {
        super.onResume();

        //will need to call start on video dom object
        Log.d(TAG, "Resuming Flex");

        setAdVisibility(true);
    }

    @Override
    public void onPause() {
        super.onPause();
        setAdVisibility(false);
    }

    /**
     * @param context             {@link Context} defining current environment
     * @param request             {@link AdRequest} for this view
     * @param config              {@link AdConfig} configuration for Ad
     * @param presentationFactory {@link PresentationFactory} capable of displaying ad in webView
     * @param listener            {@link AdContract.AdvertisementPresenter.EventListener} to listen to
     */
    public VungleBannerView(@NonNull Context context,
                            @NonNull final AdRequest request,
                            @Nullable final AdConfig config,
                            @NonNull PresentationFactory presentationFactory,
                            @NonNull final AdContract.AdvertisementPresenter.EventListener listener) {
        super(context);
        this.listener = listener;
        this.request = request;
        this.config = config;
        this.presenterFactory = presentationFactory;

        //set Hardware Acceleration
        //may negatively affect perfomance for devices under API 19
        //should for those devices: setLayerType(View.LAYER_TYPE_SOFTWARE, null);
        setLayerType(View.LAYER_TYPE_HARDWARE, null);

        // Set background transparent.
        setBackgroundColor(Color.TRANSPARENT);
        // attach ontouch listener.
        attachListeners();
    }

    private OnViewTouchListener onViewTouchListener = new OnViewTouchListener() {
        @Override
        public boolean onTouch(MotionEvent event) {
            if (presenter != null) {
                presenter.onViewTouched(event);
            }

            return false;
        }
    };

    private void attachListeners() {
        setOnTouchListener(new OnTouchListener() {
            @Override
            public boolean onTouch(View v, MotionEvent event) {
                if (onViewTouchListener != null) {
                    return onViewTouchListener.onTouch(event);
                }
                return VungleBannerView.super.onTouchEvent(event);
            }
        });
    }

    // Lint is suppressed because there is a concern of Javascript security issues below API 17
    // which is no longer supported.
    @SuppressLint({"AddJavascriptInterface", "NewApi"})
    private void prepare() {
        WebSettingsUtils.applyDefault(this);
        addJavascriptInterface(new JavascriptBridge(presenter), "Android");

        /*
            On certain devices and OS pairings (Samsung S4 etc..., Autoplay does not work)
            https://bugs.chromium.org/p/chromium/issues/detail?id=159336

            By design, auto-play is disabled.

            `loadUrl("javascript:document.getElementsByTagName('video')[0]; myvideo.start();");`
            forces videos to start on older devices.
        */
        getSettings().setMediaPlaybackRequiresUserGesture(false);
    }

    public void finishDisplayingAd() {
        finishDisplayingAdInternal(true);
    }

    public void finishDisplayingAdInternal(boolean isFinishByExternalApi) {
        if (presenter != null) {
            int flag = AdContract.AdStopReason.IS_AD_FINISHING
                    | (isFinishByExternalApi ? AdContract.AdStopReason.IS_AD_FINISHED_BY_API : 0);
            presenter.detach(flag);
        } else if (presenterFactory != null) {
            presenterFactory.destroy();
            presenterFactory = null;
            listener.onError(new VungleException(VungleException.OPERATION_CANCELED), request.getPlacementId());
        }

        if (isFinishByExternalApi) {
            SessionData.Builder sessionData = new SessionData.Builder().setEvent(SessionEvent.DISMISS_AD);
            if (request != null && request.getEventId() != null) {
                sessionData.addData(SessionAttribute.EVENT_ID, request.getEventId());
            }
            SessionTracker.getInstance().trackEvent(sessionData.build());
        }

        destroyAdView(0L);
    }

    public View renderBannerView() {
        return this;
    }

    public void setAdVisibility(boolean isVisible) {
        if (presenter != null) {
            presenter.setAdVisibility(isVisible);
        } else {
            isAdVisible.set(isVisible);
        }
    }

    @Override
    public void setPresenter(@NonNull WebAdContract.WebAdPresenter presenter) {
        //no-op
    }

    @Override
    public void setOrientation(int orientation) {
        //no-op
    }

    @Override
    public void showWebsite(@NonNull String url) {
        loadUrl(url);
    }

    @Override
    public String getWebsiteUrl() {
        return getUrl();
    }

    @Override
    public void close() {
        if (presenter != null) {
            finishDisplayingAdInternal(false);
        } else if (presenterFactory != null) {
            presenterFactory.destroy();
            presenterFactory = null;
            listener.onError(new VungleException(VungleException.OPERATION_CANCELED), request.getPlacementId());
        }
    }

    @Override
    public void destroyAdView(long webViewDestroyDelay) {
        if (destroyed)
            return;
        destroyed = true;
        presenter = null;
        presenterFactory = null;

        //can't be removed, might be not expected by developer.
        //removeWebView();
        removeJavascriptInterface("Android");
        setWebChromeClient(null);

        Runnable destroyRunnable = new Runnable() {
            @Override
            public void run() {
                stopLoading();
                setWebViewClient(null);
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                    setWebViewRenderProcessClient(null);
                }
                loadUrl("about:blank");
            }
        };
        if (webViewDestroyDelay <= 0) {
            destroyRunnable.run();
        } else {
            new HandlerScheduler().schedule(destroyRunnable, webViewDestroyDelay);
        }
    }

    @Override
    public boolean hasWebView() {
        return true;
    }

    @Override
    public void showCloseButton() {
        /// Banner does not have a close button
        throw new UnsupportedOperationException("VungleBannerView does not implement a close button");
    }

    @Override
    public void open(String deeplinkUrl, @NonNull String url, ActivityManager.LeftApplicationCallback leftApplicationCallback, PresenterAdOpenCallback adOpenCallback) {
        Log.d(TAG, "Opening " + url);
        if (!ExternalRouter.launch(deeplinkUrl, url, getContext(), leftApplicationCallback, true, adOpenCallback)) {
            Log.e(TAG, "Cannot open url " + url);
        }
    }

    @Override
    public void showDialog(@Nullable String dialogTitle,
                           @Nullable String dialogBody,
                           @NonNull String dialogContinue,
                           @NonNull String dialogClose,
                           @Nullable DialogInterface.OnClickListener responseListener) {
        /// Banner does not implement the dialog functionality
        throw new UnsupportedOperationException("VungleBannerView does not implement a dialog.");
    }

    @Override
    public void refreshDialogIfVisible() {
        //no dialog, so no-op
    }

    @Override
    public void setImmersiveMode() {
        //no-op
    }

    @Override
    public void resumeWeb() {
        onResume();
    }

    @Override
    public void pauseWeb() {
        onPause();
    }

    @Override
    public void removeWebView() {
        ViewParent viewParent = getParent();

        if (viewParent instanceof ViewManager) {
            ((ViewManager) viewParent).removeView(this);
        }
    }

    @Override
    public void updateWindow() {
        //no-op
    }

    @Override
    public void setVisibility(boolean isVisible) {
        int setVisibilityState = isVisible ? View.VISIBLE : View.INVISIBLE;
        setVisibility(setVisibilityState);
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();

        if (presenterFactory != null && presenter == null) {
            presenterFactory.getBannerViewPresentation(getContext(), request, config,
                    new CloseDelegate() {
                        @Override
                        public void close() {
                            finishDisplayingAdInternal(false);
                        }
                    },
                    new AdvertisementPresentationFactory.ViewCallback() {
                        @Override
                        public void onResult(@NonNull Pair<WebAdContract.WebAdPresenter, VungleWebClient> result, @Nullable VungleException error) {
                            presenterFactory = null;
                            if (error != null) {
                                if (listener != null) {
                                    listener.onError(error, request.getPlacementId());
                                }
                                return;
                            }

                            presenter = result.first;
                            VungleWebClient webViewClient = result.second;
                            setWebViewClient(webViewClient);

                            presenter.setEventListener(listener);
                            presenter.attach(VungleBannerView.this, null);
                            prepare();
                            if (isAdVisible.get() != null) {
                                setAdVisibility(isAdVisible.get());
                            }
                            ViewGroup.LayoutParams layoutParams = getLayoutParams();
                            if (layoutParams != null) {
                                layoutParams.height = ViewGroup.LayoutParams.MATCH_PARENT;
                                layoutParams.width = ViewGroup.LayoutParams.MATCH_PARENT;
                            }
                        }
                    });

        }

        broadcastReceiver = new BroadcastReceiver() {
            @Override
            public void onReceive(Context context, Intent intent) {
                String command = intent.getStringExtra(AdContract.AdvertisementBus.COMMAND);
                if (AdContract.AdvertisementBus.STOP_ALL.equalsIgnoreCase(command)) {
                    finishDisplayingAdInternal(false);
                } else {
                    VungleLogger.warn(VungleBannerView.class.getSimpleName() + "#onAttachedToWindow", String.format("Receiving Invalid Broadcast: %1$s", command));
                }
            }
        };

        LocalBroadcastManager.getInstance(getContext()).registerReceiver(broadcastReceiver, new IntentFilter(AdContract.AdvertisementBus.ACTION));

        resumeWeb();
    }

    @Override
    public void onWindowFocusChanged(boolean hasWindowFocus) {
        super.onWindowFocusChanged(hasWindowFocus);
        setAdVisibility(hasWindowFocus);
    }

    @Override
    protected void onDetachedFromWindow() {
        LocalBroadcastManager.getInstance(getContext()).unregisterReceiver(broadcastReceiver);
        super.onDetachedFromWindow();
        if (presenterFactory != null) {
            presenterFactory.destroy();
        }

        pauseWeb();
    }
}