package com.vungle.warren.utility;

import android.app.Activity;
import android.app.Application;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;

import java.lang.ref.WeakReference;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.vungle.warren.ui.PresenterAdOpenCallback;

public class ActivityManager implements Application.ActivityLifecycleCallbacks {

    public static final String TAG = ActivityManager.class.getSimpleName();

    public static class LifeCycleCallback {
        public void onStart() {
            //no-op
        }

        public void onStop() {
            //no-op
        }

        public void onResume() {
            //no-op
        }

        public void onPause() {
            //no-op
        }
    }

    public interface LeftApplicationCallback {
        void onLeftApplication();
    }

    private static final ActivityManager instance = new ActivityManager();

    private boolean initialized;
    private int started;
    private int resumed;
    private CopyOnWriteArraySet<LifeCycleCallback> callbacks = new CopyOnWriteArraySet<>();
    private ConcurrentHashMap<LeftApplicationCallback, LifeCycleCallback> adLeftCallbacks = new ConcurrentHashMap<>();

    @VisibleForTesting
    static final long TIMEOUT = 3000;//no activity to show timeout
    //handle configuration changes
    private Handler handler;
    private boolean paused = true;
    private boolean stopped = true;
    @VisibleForTesting
    static final long CONFIG_CHANGE_DELAY = 700;//ms, max render delay, must be less than TIMEOUT
    private Runnable configChangeRunnable = new Runnable() {
        @Override
        public void run() {
            if (resumed == 0 && !paused) {
                paused = true;
                for (LifeCycleCallback callback : callbacks) {
                    callback.onPause();
                }
            }
            if (started == 0 && paused && !stopped) {
                stopped = true;
                for (LifeCycleCallback callback : callbacks) {
                    callback.onStop();
                }
            }
        }
    };

    private ActivityManager() {
        //no-op
    }

    public static ActivityManager getInstance() {
        return instance;
    }

    public boolean isInitialized() {
        return initialized;
    }

    public void init(Context context) {
        if (initialized)
            return;

        handler = new Handler(Looper.getMainLooper());
        Application app = (Application) context.getApplicationContext();
        app.registerActivityLifecycleCallbacks(this);
        initialized = true;
    }

    @VisibleForTesting
    void deInit(Context context) {
        Application app = (Application) context.getApplicationContext();
        app.unregisterActivityLifecycleCallbacks(this);
        started = 0;
        resumed = 0;
        paused = true;
        stopped = true;
        initialized = false;
        callbacks.clear();
        adLeftCallbacks.clear();
    }

    public static void startWhenForeground(final Context context, @Nullable final Intent deepLinkOverrideIntent,
                                           final Intent defaultIntent, @Nullable final LeftApplicationCallback leftCallback, @Nullable final PresenterAdOpenCallback adOpenCallback) {
        final WeakReference<Context> weakContext = new WeakReference<>(context);
        if (instance.inForeground()) {
            if (startActivityHandleException(context, deepLinkOverrideIntent, defaultIntent, adOpenCallback)) {
                instance.addOnNextAppLeftCallback(leftCallback);
            }
        } else {
            instance.addListener(new LifeCycleCallback() {
                @Override
                public void onStart() {
                    super.onStart();
                    instance.removeListener(this);
                    Context context = weakContext.get();
                    if (context != null && startActivityHandleException(context, deepLinkOverrideIntent, defaultIntent, adOpenCallback)) {
                        instance.addOnNextAppLeftCallback(leftCallback);
                    }
                }

            });
        }
    }

    public static void startWhenForeground(final Context context, @Nullable final Intent deeplinkOverrideIntent,
                                           Intent defaultIntent, @Nullable final LeftApplicationCallback leftCallback) {
        startWhenForeground(context, deeplinkOverrideIntent, defaultIntent, leftCallback, null);
    }

    @VisibleForTesting
    protected boolean inForeground() {
        return !initialized || (started > 0);
    }

    public void addListener(LifeCycleCallback callback) {
        callbacks.add(callback);
    }

    private void removeListener(LifeCycleCallback callback) {
        callbacks.remove(callback);
    }

    @Override
    public void onActivityCreated(@NonNull Activity activity, Bundle savedInstanceState) {
        //no-op
    }

    @Override
    public void onActivityStarted(@NonNull Activity activity) {
        started++;
        if (started == 1 && stopped) {
            stopped = false;
            for (LifeCycleCallback callback : callbacks) {
                callback.onStart();
            }
        }
    }

    @Override
    public void onActivityStopped(@NonNull Activity activity) {
        started = Math.max(0, started - 1);
        handler.postDelayed(configChangeRunnable, CONFIG_CHANGE_DELAY);
    }

    @Override
    public void onActivityResumed(@NonNull Activity activity) {
        resumed++;
        if (resumed == 1) {
            if (paused) {
                paused = false;
                for (LifeCycleCallback callback : callbacks) {
                    callback.onResume();
                }
            } else {
                handler.removeCallbacks(configChangeRunnable);
            }
        }
    }

    @Override
    public void onActivityPaused(@NonNull Activity activity) {
        resumed = Math.max(0, resumed - 1);
        handler.postDelayed(configChangeRunnable, CONFIG_CHANGE_DELAY);
    }

    @Override
    public void onActivitySaveInstanceState(@NonNull Activity activity, @NonNull Bundle outState) {
        //no-op
    }

    @Override
    public void onActivityDestroyed(@NonNull Activity activity) {
        //no-op
    }

    public void addOnNextAppLeftCallback(@Nullable LeftApplicationCallback leftCallback) {
        if (leftCallback == null) {
            return;
        }

        if (!initialized) {
            leftCallback.onLeftApplication();
            return;
        }

        final WeakReference<LeftApplicationCallback> weakCallback = new WeakReference<>(leftCallback);
        final Runnable cancelRunnable = new Runnable() {
            @Override
            public void run() {
                handler.removeCallbacks(this);
                removeOnNextAppLeftCallback(weakCallback.get());
            }
        };

        LifeCycleCallback callback = new LifeCycleCallback() {
            boolean wasPaused = false;

            @Override
            public void onStop() {
                super.onStop();
                LeftApplicationCallback leftCallback = weakCallback.get();
                if (wasPaused && leftCallback != null && adLeftCallbacks.containsKey(leftCallback)) {
                    leftCallback.onLeftApplication();
                }
                removeOnNextAppLeftCallback(leftCallback);
                handler.removeCallbacks(cancelRunnable);
            }

            @Override
            public void onResume() {
                super.onResume();
                handler.postDelayed(cancelRunnable, CONFIG_CHANGE_DELAY * 2);//wait for delayed onPause
            }

            @Override
            public void onPause() {
                super.onPause();
                wasPaused = true;
                handler.removeCallbacks(cancelRunnable);
            }
        };

        //handle starting from background and no activity resolved
        adLeftCallbacks.put(leftCallback, callback);
        if (inForeground()) {
            handler.postDelayed(cancelRunnable, TIMEOUT);
            addListener(callback);
        } else {
            getInstance().addListener(new LifeCycleCallback() {
                @Override
                public void onStart() {
                    instance.removeListener(this);
                    LifeCycleCallback callback = adLeftCallbacks.get(weakCallback.get());
                    if (callback != null) {
                        handler.postDelayed(cancelRunnable, TIMEOUT);
                        addListener(callback);
                    }
                }
            });
        }
    }

    private void removeOnNextAppLeftCallback(@Nullable LeftApplicationCallback leftCallback) {
        if (leftCallback == null)
            return;
        LifeCycleCallback callback = adLeftCallbacks.remove(leftCallback);
        if (callback != null) {
            removeListener(callback);
        }
    }

    private static boolean startActivityHandleException(final Context context, @Nullable final Intent deepLinkOverrideIntent, Intent defaultIntent, PresenterAdOpenCallback adOpenCallback) {
        if (deepLinkOverrideIntent == null && defaultIntent == null) {
            return false;
        }
        try {
            if (deepLinkOverrideIntent != null) {
                context.startActivity(deepLinkOverrideIntent);
            } else {
                context.startActivity(defaultIntent);
            }

        } catch (ActivityNotFoundException actNotFoundException) {
            Log.e(TAG, "Cannot find activity to handle the Implicit intent: " + actNotFoundException.getLocalizedMessage());
            try {
                if (deepLinkOverrideIntent == null || defaultIntent == null) {
                    return false;
                }
                context.startActivity(defaultIntent);
            } catch (ActivityNotFoundException exception) {
                return false;
            }
            if (adOpenCallback != null) {
                adOpenCallback.onAdOpenType(PresenterAdOpenCallback.AdOpenType.DEFAULT);
            }
            return true;
        }
        if (adOpenCallback != null) {
            adOpenCallback.onAdOpenType(deepLinkOverrideIntent != null ? PresenterAdOpenCallback.AdOpenType.DEEP_LINK :
                    PresenterAdOpenCallback.AdOpenType.DEFAULT);
        }
        return true;
    }
}
