package com.vungle.warren.utility;

import android.util.Base64;

import androidx.annotation.Nullable;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;
import com.vungle.warren.VungleLogger;
import com.vungle.warren.model.admarkup.AdMarkup;
import com.vungle.warren.model.admarkup.AdMarkupV1;
import com.vungle.warren.model.admarkup.AdMarkupV2;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.zip.GZIPInputStream;

public class AdMarkupDecoder {

    public static @Nullable
    AdMarkup decode(@Nullable String adMarkup) {
        if (adMarkup == null) return null;

        try {
            JsonElement jsonElement = JsonParser.parseString(adMarkup);
            if (!jsonElement.isJsonObject()) {
                return null;
            }
            JsonObject jsonObject = jsonElement.getAsJsonObject();
            int version = jsonElement.getAsJsonObject().get("version").getAsInt();
            switch (version) {
                case 1:
                    return AdMarkupV1.fromString(adMarkup);
                case 2:
                    return serializeAdMarkupV2(jsonObject);
            }
        } catch (JsonSyntaxException e) {
            logError();
            return null;
        }
        return null;
    }

    private static AdMarkupV2 serializeAdMarkupV2(JsonObject jsonObject) {
        String payload = jsonObject.get("adunit").getAsString();
        JsonArray impressionsJsonArray = jsonObject.get("impression").getAsJsonArray();
        String[] impressions = new String[impressionsJsonArray.size()];

        for (int i = 0; i < impressionsJsonArray.size(); i++) {
            impressions[i] = impressionsJsonArray.get(i).getAsString();
        }
        byte[] compressed = Base64.decode(payload, Base64.DEFAULT);
        
        try {
            String decoded = gzipDecode(compressed);
            JsonObject adUnit = JsonParser.parseString(decoded).getAsJsonObject();
            return new AdMarkupV2(adUnit, impressions);
        } catch (IOException e) {
            logError();
        }
        return null;
    }

    private static String gzipDecode(byte[] compressed) throws IOException {
        final int BUFFER_SIZE = 32;
        ByteArrayInputStream is = new ByteArrayInputStream(compressed);
        GZIPInputStream gis = new GZIPInputStream(is, BUFFER_SIZE);
        StringBuilder string = new StringBuilder();
        byte[] data = new byte[BUFFER_SIZE];
        int bytesRead;
        while ((bytesRead = gis.read(data)) != -1) {
            string.append(new String(data, 0, bytesRead));
        }
        gis.close();
        is.close();
        return string.toString();
    }

    private static void logError() {
        VungleLogger.error(
                AdMarkupDecoder.class.getName(),
                "Encountered issue serializing models"
        );
    }
}
