package com.vungle.warren.utility;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.text.TextUtils;
import android.util.Log;
import android.util.LruCache;

import java.util.concurrent.Executor;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import static com.vungle.warren.model.Advertisement.FILE_SCHEME;

/**
 * ImageLoader for native ad assets and only for already cached local files currently.
 */
public class ImageLoader {

    private static final String TAG = ImageLoader.class.getSimpleName();

    private static final ImageLoader sInstance = new ImageLoader();

    private final LruCache<String, Bitmap> lruCache;
    private Executor ioExecutor;

    private ImageLoader() {
        int maxMemory = (int) (Runtime.getRuntime().maxMemory() / 1024);
        // Use 1/8th of the available memory for this memory cache.
        int cacheSize = maxMemory / 8;
        lruCache = new LruCache<String, Bitmap>(cacheSize) {
            @Override
            protected int sizeOf(String key, Bitmap value) {
                return value.getByteCount() / 1024;
            }
        };
    }

    @VisibleForTesting
    ImageLoader(LruCache<String, Bitmap> lruCache) {
        this.lruCache = lruCache;
    }

    public static ImageLoader getInstance() {
        return sInstance;
    }

    public void init(@NonNull Executor ioExecutor) {
        this.ioExecutor = ioExecutor;
    }

    public interface ImageLoaderListener {
        void onImageLoaded(Bitmap bitmap);
    }

    public void displayImage(@Nullable final String uri, @Nullable final ImageLoaderListener listener) {
        if (ioExecutor == null) {
            Log.w(TAG, "ImageLoader not initialized.");
            return;
        }

        if (TextUtils.isEmpty(uri)) {
            Log.w(TAG, "the uri is required.");
            return;
        }

        ioExecutor.execute(new Runnable() {
            @Override
            public void run() {
                if (!uri.startsWith(FILE_SCHEME)) {
                    return;
                }

                Bitmap cachedBitmap = lruCache.get(uri);
                if (cachedBitmap != null && !cachedBitmap.isRecycled()) {
                    if (listener != null) {
                        listener.onImageLoaded(cachedBitmap);
                    }
                    return;
                }

                String assetFile = uri.substring(FILE_SCHEME.length());
                Bitmap bitmap = BitmapFactory.decodeFile(assetFile);
                if (bitmap != null) {
                    lruCache.put(uri, bitmap);
                    if (listener != null) {
                        listener.onImageLoaded(bitmap);
                    }
                } else {
                    Log.w(TAG, "decode bitmap failed.");
                }
            }
        });
    }
}
