package com.vungle.warren.utility;

import android.app.Activity;
import android.content.Context;
import android.graphics.Rect;
import android.os.Handler;
import android.util.Log;
import android.view.View;
import android.view.ViewTreeObserver;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Map;
import java.util.WeakHashMap;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import static android.view.ViewTreeObserver.OnPreDrawListener;

/**
 * Impression tracker to determine when view become visible or invisible.
 */
public class ImpressionTracker {

    private static final String TAG = ImpressionTracker.class.getSimpleName();

    // The minimum percent of the ad to be on screen
    private static final int MIN_VISIBILITY_PERCENTAGE = 1;

    // Time interval to use for throttling visibility checks.
    private static final int VISIBILITY_THROTTLE_MILLIS = 100;

    // A rect to use for hit testing. Create this once to avoid excess garbage collection
    private final Rect clipRect = new Rect();

    // Listener that passes the view when a visibility check occurs
    public interface ImpressionListener {
        void onImpression(View view);
    }

    @NonNull
    @VisibleForTesting
    final OnPreDrawListener onPreDrawListener;
    @NonNull
    @VisibleForTesting
    WeakReference<ViewTreeObserver> weakViewTreeObserver;

    @VisibleForTesting
    static class TrackingInfo {
        int minViewablePercent;
        ImpressionListener impressionListener;
    }

    // Views that are being tracked, mapped to the min viewable percentage
    @NonNull
    private final Map<View, TrackingInfo> trackedViews;

    // Runnable to run on each visibility loop
    @NonNull
    private final VisibilityRunnable visibilityRunnable;

    // Handler for visibility
    @NonNull
    private final Handler visibilityHandler;

    // Whether the visibility runnable is scheduled
    private boolean isVisibilityScheduled;

    public ImpressionTracker(@NonNull final Context context) {
        this(context, new WeakHashMap<View, TrackingInfo>(10), new Handler());
    }

    @VisibleForTesting
    ImpressionTracker(@NonNull final Context context,
                      @NonNull final Map<View, TrackingInfo> trackedViews,
                      @NonNull final Handler visibilityHandler) {
        this.trackedViews = trackedViews;
        this.visibilityHandler = visibilityHandler;
        visibilityRunnable = new VisibilityRunnable();

        onPreDrawListener = new OnPreDrawListener() {
            @Override
            public boolean onPreDraw() {
                scheduleVisibilityCheck();
                return true;
            }
        };

        weakViewTreeObserver = new WeakReference<>(null);
        setViewTreeObserver(context, null);
    }

    private void setViewTreeObserver(@Nullable final Context context, @Nullable final View view) {
        final ViewTreeObserver originalViewTreeObserver = weakViewTreeObserver.get();
        if (originalViewTreeObserver != null && originalViewTreeObserver.isAlive()) {
            return;
        }

        final View rootView = getTopView(context, view);
        if (rootView == null) {
            Log.d(TAG, "Unable to set ViewTreeObserver due to no available root view.");
            return;
        }

        final ViewTreeObserver viewTreeObserver = rootView.getViewTreeObserver();
        if (!viewTreeObserver.isAlive()) {
            Log.d(TAG, "The root view tree observer was not alive");
            return;
        }

        weakViewTreeObserver = new WeakReference<>(viewTreeObserver);
        viewTreeObserver.addOnPreDrawListener(onPreDrawListener);
    }

    /**
     * Tracks the given view for impression.
     */
    public void addView(@NonNull final View view, @Nullable ImpressionListener listener) {
        setViewTreeObserver(view.getContext(), view);

        // Find the view if already tracked
        TrackingInfo trackingInfo = trackedViews.get(view);
        if (trackingInfo == null) {
            trackingInfo = new TrackingInfo();
            trackedViews.put(view, trackingInfo);
            scheduleVisibilityCheck();
        }

        trackingInfo.minViewablePercent = MIN_VISIBILITY_PERCENTAGE;
        trackingInfo.impressionListener = listener;
    }

    @VisibleForTesting
    void removeView(@NonNull final View view) {
        trackedViews.remove(view);
    }

    public void clear() {
        trackedViews.clear();
        visibilityHandler.removeMessages(0);
        isVisibilityScheduled = false;
    }

    /**
     * Destroy the impression tracker.
     */
    public void destroy() {
        clear();
        final ViewTreeObserver viewTreeObserver = weakViewTreeObserver.get();
        if (viewTreeObserver != null && viewTreeObserver.isAlive()) {
            viewTreeObserver.removeOnPreDrawListener(onPreDrawListener);
        }
        weakViewTreeObserver.clear();
    }

    @Nullable
    private View getTopView(@Nullable Context context, @Nullable View view) {
        View topView = null;
        if (context instanceof Activity) {
            topView = ((Activity) context).getWindow().getDecorView().findViewById(android.R.id.content);
        }

        if (topView == null && view != null) {
            View rootView = view.getRootView();
            if (rootView != null) {
                topView = rootView.findViewById(android.R.id.content);
            }
        }

        return topView;
    }

    private void scheduleVisibilityCheck() {
        if (isVisibilityScheduled) {
            return;
        }

        isVisibilityScheduled = true;
        visibilityHandler.postDelayed(visibilityRunnable, VISIBILITY_THROTTLE_MILLIS);
    }

    @VisibleForTesting
    class VisibilityRunnable implements Runnable {

        @NonNull
        private final ArrayList<View> mVisibleViews;

        VisibilityRunnable() {
            mVisibleViews = new ArrayList<>();
        }

        @Override
        public void run() {
            isVisibilityScheduled = false;
            for (final Map.Entry<View, TrackingInfo> entry : trackedViews.entrySet()) {
                final View view = entry.getKey();
                final int minPercentageViewed = entry.getValue().minViewablePercent;

                if (isVisible(view, minPercentageViewed)) {
                    mVisibleViews.add(view);
                }
            }

            for (View view : mVisibleViews) {
                TrackingInfo info = trackedViews.get(view);
                if (info != null && info.impressionListener != null) {
                    info.impressionListener.onImpression(view);
                }

                removeView(view);
            }

            mVisibleViews.clear();
        }
    }

    /**
     * Whether the view is at least certain amount visible.
     */
    private boolean isVisible(@Nullable final View view, final int minPercentageViewed) {
        if (view == null || view.getVisibility() != View.VISIBLE || view.getParent() == null) {
            return false;
        }

        if (!view.getGlobalVisibleRect(clipRect)) {
            return false;
        }

        final long visibleViewArea = (long) clipRect.height() * clipRect.width();
        final long totalViewArea = (long) view.getHeight() * view.getWidth();

        if (totalViewArea <= 0) {
            return false;
        }

        return 100 * visibleViewArea >= minPercentageViewed * totalViewArea;
    }
}
