package com.vungle.warren.utility;

import android.Manifest;
import android.annotation.SuppressLint;
import android.content.Context;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkInfo;
import android.net.NetworkRequest;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;

import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.atomic.AtomicInteger;

import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.core.content.PermissionChecker;

public class NetworkProvider {

    private static final String TAG = NetworkProvider.class.getSimpleName();
    private static NetworkProvider INSTANCE;
    private final Context ctx;
    @Nullable
    private final ConnectivityManager cm;
    private final AtomicInteger currentNetwork = new AtomicInteger();
    private ConnectivityManager.NetworkCallback networkCallback;
    private final Set<NetworkListener> listeners = new CopyOnWriteArraySet<>();
    public static final int TYPE_NONE = -1;
    private boolean enabled;
    private final Handler handler = new Handler(Looper.getMainLooper());
    static final long NETWORK_CHECK_DELAY = 30L * 1000L;


    private NetworkProvider(Context context) {
        ctx = context.getApplicationContext();
        cm = (ConnectivityManager) ctx.getSystemService(Context.CONNECTIVITY_SERVICE);
        currentNetwork.set(getCurrentNetworkType());
    }

    public static synchronized NetworkProvider getInstance(Context context) {
        if (INSTANCE == null) {
            INSTANCE = new NetworkProvider(context);
        }

        return INSTANCE;
    }

    @SuppressLint("newApi")
    private synchronized void setEnableNetworkListener(boolean enabled) {
        if (this.enabled == enabled)
            return;
        this.enabled = enabled;
        if (cm != null) {
            try {
                if (enabled) {
                    NetworkRequest.Builder builder = new NetworkRequest.Builder();
                    builder.addCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET);
                    cm.registerNetworkCallback(builder.build(), getNetworkCallback());
                } else {
                    cm.unregisterNetworkCallback(getNetworkCallback());
                }
            } catch (Exception ex) {
                Log.e(TAG, ex.getMessage());
            }
        }
    }

    @SuppressLint("newApi")
    private ConnectivityManager.NetworkCallback getNetworkCallback() {
        return networkCallback != null ? networkCallback : (networkCallback = new ConnectivityManager.NetworkCallback() {

            @Override
            @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
            public void onAvailable(Network network) {
                super.onAvailable(network);
                onNetworkChanged();
            }

            @Override
            @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
            public void onLost(Network network) {
                super.onLost(network);
                onNetworkChanged();
            }
        });
    }

    private void postToListeners(final int currentNetwork) {
        handler.post(new Runnable() {
            @Override
            public void run() {
                for (NetworkListener listener : listeners) {
                    listener.onChanged(currentNetwork);
                }
            }
        });
    }

    public void onNetworkChanged() {
        getCurrentNetworkType();
    }

    public int getCurrentNetworkType() {
        if (cm == null || PermissionChecker.checkCallingOrSelfPermission(ctx,
                Manifest.permission.ACCESS_NETWORK_STATE) != PermissionChecker.PERMISSION_GRANTED) {

            currentNetwork.set(TYPE_NONE);
            return TYPE_NONE;
        }

        NetworkInfo activeNetwork = cm.getActiveNetworkInfo();

        int newNetwork = activeNetwork != null && activeNetwork.isConnectedOrConnecting()
                ? activeNetwork.getType()
                : TYPE_NONE;

        int old = currentNetwork.getAndSet(newNetwork);

        if (newNetwork != old) {
            Log.d(TAG, "on network changed: " + old + "->" + newNetwork);
            postToListeners(newNetwork);
        }

        setEnableNetworkListener(!listeners.isEmpty());
        return newNetwork;
    }

    public void addListener(NetworkListener networkListener) {
        listeners.add(networkListener);
        setEnableNetworkListener(true);
    }

    public void removeListener(NetworkListener networkListener) {
        listeners.remove(networkListener);
        setEnableNetworkListener(!listeners.isEmpty());
    }

    private Runnable typeRunnable = new Runnable() {
        @Override
        public void run() {
            if (!listeners.isEmpty()) {
                onNetworkChanged();
                handler.postDelayed(typeRunnable, NETWORK_CHECK_DELAY);
            }
        }
    };

    public interface NetworkListener {

        void onChanged(int type);

    }
}
