package com.vungle.warren.utility;

import android.os.Handler;
import android.os.Looper;

import java.util.List;
import java.util.concurrent.AbstractExecutorService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.TimeUnit;

import androidx.annotation.NonNull;

public class SDKExecutors implements Executors {

    private static final int SINGLE_CORE_POOL_SIZE = 1;

    private static final int IO_KEEP_ALIVE_TIME_SECONDS = 5;
    private static final int VUNGLE_KEEP_ALIVE_TIME_SECONDS = 10;

    private static final int JOBS_KEEP_ALIVE_TIME_SECONDS = 1;

    private static int NUMBER_OF_CORES = Runtime.getRuntime().availableProcessors();

    /**
     * Executor to execute IO file, db calls.
     */
    private static final VungleThreadPoolExecutor IO_EXECUTOR;
    /**
     * Executor to execute business logic and wait for results from other executors.
     */
    private static final VungleThreadPoolExecutor BACKGROUND_EXECUTOR;
    /**
     * Executor to execute tasks and jobs.
     */
    private static final VungleThreadPoolExecutor JOB_EXECUTOR;
    /**
     * Main/UI thread executor implementation.
     */
    private static final ExecutorService UI_EXECUTOR;

    /**
     * Executor to execute public blocking Vungle API calls as fast as possible.
     */
    private static final VungleThreadPoolExecutor API_EXECUTOR;

    /**
     * Executor to execute logging IO file.
     */
    private static final VungleThreadPoolExecutor LOGGER_EXECUTOR;

    /**
     * Executor to execute AsyncTasks and probably similar things on
     */
    private static final VungleThreadPoolExecutor TASK_EXECUTOR;

    /**
     * Executor to execute long task to get user agent
     */
    private static final VungleThreadPoolExecutor UA_EXECUTOR;

    /**
     * Executor to execute AssetDownloader tasks.
     */
    private static final VungleThreadPoolExecutor DOWNLOADER_EXECUTOR;

    /**
     * Executor for offloading some work from main thread maintaining higher priority
     */
    private static final VungleThreadPoolExecutor OFFLOAD_EXECUTOR;

    /**
     * Executor for sending and tracking events for Session Data
     */
    private static final VungleThreadPoolExecutor SESSION_DATA_EXECUTOR;


    static {
        UI_EXECUTOR = new AbstractExecutorService() {

            private final Handler UIHandler = new Handler(Looper.getMainLooper());

            @Override
            public void shutdown() {
                //no-op
            }


            @Override
            public List<Runnable> shutdownNow() {
                return null;
            }

            @Override
            public boolean isShutdown() {
                return false;
            }

            @Override
            public boolean isTerminated() {
                return false;
            }

            @Override
            public boolean awaitTermination(long timeout, @NonNull TimeUnit unit) {
                return false;
            }

            @Override
            public void execute(@NonNull Runnable command) {
                UIHandler.post(command);
            }

        };

        JOB_EXECUTOR = new VungleThreadPoolExecutor(
                NUMBER_OF_CORES,
                NUMBER_OF_CORES,
                JOBS_KEEP_ALIVE_TIME_SECONDS,
                TimeUnit.SECONDS,
                new PriorityBlockingQueue<Runnable>(),
                new NamedThreadFactory("vng_jr")
        );

        IO_EXECUTOR = new VungleThreadPoolExecutor(
                SINGLE_CORE_POOL_SIZE,
                SINGLE_CORE_POOL_SIZE,
                IO_KEEP_ALIVE_TIME_SECONDS,
                TimeUnit.SECONDS,
                new LinkedBlockingQueue<Runnable>(),
                new NamedThreadFactory("vng_io")
        );

        LOGGER_EXECUTOR = new VungleThreadPoolExecutor(
                SINGLE_CORE_POOL_SIZE,
                SINGLE_CORE_POOL_SIZE,
                IO_KEEP_ALIVE_TIME_SECONDS,
                TimeUnit.SECONDS,
                new LinkedBlockingQueue<Runnable>(),
                new NamedThreadFactory("vng_logger")
        );

        BACKGROUND_EXECUTOR = new VungleThreadPoolExecutor(
                SINGLE_CORE_POOL_SIZE,
                SINGLE_CORE_POOL_SIZE,
                VUNGLE_KEEP_ALIVE_TIME_SECONDS,
                TimeUnit.SECONDS,
                new LinkedBlockingQueue<Runnable>(),
                new NamedThreadFactory("vng_background")
        );

        API_EXECUTOR = new VungleThreadPoolExecutor(
                SINGLE_CORE_POOL_SIZE,
                SINGLE_CORE_POOL_SIZE,
                VUNGLE_KEEP_ALIVE_TIME_SECONDS,
                TimeUnit.SECONDS,
                new LinkedBlockingQueue<Runnable>(),
                new NamedThreadFactory("vng_api")
        );

        TASK_EXECUTOR = new VungleThreadPoolExecutor(
                SINGLE_CORE_POOL_SIZE,
                20,
                VUNGLE_KEEP_ALIVE_TIME_SECONDS,
                TimeUnit.SECONDS,
                new SynchronousQueue<Runnable>(),
                new NamedThreadFactory("vng_task")
        );

        UA_EXECUTOR = new VungleThreadPoolExecutor(
                SINGLE_CORE_POOL_SIZE,
                SINGLE_CORE_POOL_SIZE,
                VUNGLE_KEEP_ALIVE_TIME_SECONDS,
                TimeUnit.SECONDS,
                new LinkedBlockingQueue<Runnable>(),
                new NamedThreadFactory("vng_ua")
        );

        DOWNLOADER_EXECUTOR = new VungleThreadPoolExecutor(
                4,
                4,
                1,
                TimeUnit.SECONDS,
                new PriorityBlockingQueue<Runnable>(),
                new NamedThreadFactory("vng_down")
        );

        OFFLOAD_EXECUTOR = new VungleThreadPoolExecutor(
                SINGLE_CORE_POOL_SIZE,
                SINGLE_CORE_POOL_SIZE,
                VUNGLE_KEEP_ALIVE_TIME_SECONDS,
                TimeUnit.SECONDS,
                new LinkedBlockingQueue<Runnable>(),
                new NamedThreadFactory("vng_ol")
        );

        SESSION_DATA_EXECUTOR = new VungleThreadPoolExecutor(
                SINGLE_CORE_POOL_SIZE,
                SINGLE_CORE_POOL_SIZE,
                IO_KEEP_ALIVE_TIME_SECONDS,
                TimeUnit.SECONDS,
                new LinkedBlockingQueue<Runnable>(),
                new NamedThreadFactory("vng_session")
        );
    }

    @Override
    public ExecutorService getUIExecutor() {
        return UI_EXECUTOR;
    }

    @Override
    public VungleThreadPoolExecutor getBackgroundExecutor() {
        return BACKGROUND_EXECUTOR;
    }

    @Override
    public VungleThreadPoolExecutor getIOExecutor() {
        return IO_EXECUTOR;
    }

    @Override
    public VungleThreadPoolExecutor getJobExecutor() {
        return JOB_EXECUTOR;
    }

    @Override
    public VungleThreadPoolExecutor getApiExecutor() {
        return API_EXECUTOR;
    }

    @Override
    public VungleThreadPoolExecutor getLoggerExecutor() {
        return LOGGER_EXECUTOR;
    }

    @Override
    public VungleThreadPoolExecutor getTaskExecutor() {
        return TASK_EXECUTOR;
    }

    @Override
    public VungleThreadPoolExecutor getOffloadExecutor() {
        return OFFLOAD_EXECUTOR;
    }

    @Override
    public VungleThreadPoolExecutor getUAExecutor() {
        return UA_EXECUTOR;
    }

    @Override
    public VungleThreadPoolExecutor getDownloaderExecutor() {
        return DOWNLOADER_EXECUTOR;
    }

    @Override
    public VungleThreadPoolExecutor getSessionDataExecutor() {
        return SESSION_DATA_EXECUTOR;
    }

}
