package com.vungle.warren.utility.platform;

import static android.os.Build.VERSION_CODES.LOLLIPOP;
import static com.vungle.warren.model.Cookie.APP_SET_ID;
import static com.vungle.warren.model.Cookie.APP_SET_ID_COOKIE;
import static com.vungle.warren.model.Cookie.USER_AGENT_ID_COOKIE;

import android.Manifest;
import android.annotation.SuppressLint;
import android.content.ContentResolver;
import android.content.Context;
import android.content.pm.PackageManager;
import android.media.AudioManager;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Environment;
import android.os.PowerManager;
import android.provider.Settings;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.util.Consumer;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.appset.AppSet;
import com.google.android.gms.appset.AppSetIdClient;
import com.google.android.gms.appset.AppSetIdInfo;
import com.google.android.gms.common.GooglePlayServicesNotAvailableException;
import com.google.android.gms.tasks.OnSuccessListener;
import com.google.android.gms.tasks.Task;
import com.vungle.warren.model.AdvertisingInfo;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.utility.TimeoutProvider;
import com.vungle.warren.utility.VungleThreadPoolExecutor;

import java.util.concurrent.TimeUnit;

public class AndroidPlatform implements Platform {

    private final PowerManager powerManager;
    private final Context context;
    private final Repository repository;

    private final VungleThreadPoolExecutor uaExecutor;
    private final String TAG = AndroidPlatform.class.getSimpleName();
    private final TimeoutProvider timeoutProvider;
    private String appSetId;
    private AdvertisingInfo advertisingInfo = null;

    /**
     * Andorid ID Collection Disabled as Fallback
     */
    private boolean defaultIdFallbackDisabled;

    public AndroidPlatform(Context context, Repository repository, VungleThreadPoolExecutor uaExecutor, TimeoutProvider timeoutProvider) {
        this.context = context;
        this.powerManager = (PowerManager) context.getSystemService(Context.POWER_SERVICE);
        this.repository = repository;
        this.uaExecutor = uaExecutor;
        this.timeoutProvider = timeoutProvider;
        //as this value will rarely change ( only when user has deleted all apps from same developer)
        //we can call this only once during session , persist it in Cookie for future and use runtime value
        updateAppSetID();
    }

    @Override
    public boolean isAtLeastMinimumSDK() {
        return Build.VERSION.SDK_INT >= LOLLIPOP;
    }

    @Override
    public boolean getIsBatterySaverEnabled() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            return powerManager.isPowerSaveMode();
        }
        return false;
    }

    @Override
    public boolean getIsSideloaded() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            if (PackageManager.PERMISSION_GRANTED ==
                    context.checkCallingOrSelfPermission(Manifest.permission.REQUEST_INSTALL_PACKAGES)) {
                return context.getApplicationContext().getPackageManager().canRequestPackageInstalls();
            }
        } else {
            try {
                return Settings.Secure.getInt(context.getContentResolver(), Settings.Secure.INSTALL_NON_MARKET_APPS) == 1;
            } catch (Settings.SettingNotFoundException e) {
                return false;
            }
        }
        return false;
    }

    @Override
    public double getVolumeLevel() {
        AudioManager audioManager = (AudioManager) context.getSystemService(Context.AUDIO_SERVICE);
        double max = audioManager.getStreamMaxVolume(AudioManager.STREAM_MUSIC);
        double current = audioManager.getStreamVolume(AudioManager.STREAM_MUSIC);
        return current / max;
    }

    @Override
    public boolean getIsSoundEnabled() {
        AudioManager audioManager = (AudioManager) context.getSystemService(Context.AUDIO_SERVICE);
        int current = audioManager.getStreamVolume(AudioManager.STREAM_MUSIC);
        return current > 0;
    }

    @Override
    public boolean getIsSDCardPresent() {
        return Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED);
    }

    @Override
    public void getUserAgentLazy(final Consumer<String> consumer) {
        uaExecutor.execute(new Runnable() {
            @Override
            public void run() {
                new WebViewUtil(context, repository).getUserAgent(consumer);
            }
        });
    }

    @SuppressLint({"HardwareIds", "NewApi"})
    @Override
    @NonNull
    public AdvertisingInfo getAdvertisingInfo() {
        if (advertisingInfo != null && !TextUtils.isEmpty(advertisingInfo.advertisingId)) {
            return advertisingInfo;
        }
        advertisingInfo = new AdvertisingInfo();
        try {
            AdvertisingIdClient.Info idInfo;
            if (MANUFACTURER_AMAZON.equals(Build.MANUFACTURER)) {
                try {
                    ContentResolver cr = context.getContentResolver();
                    // load user's tracking preference
                    advertisingInfo.limitAdTracking = (Settings.Secure.getInt(cr, "limit_ad_tracking") == 1);
                    // load advertising
                    advertisingInfo.advertisingId = Settings.Secure.getString(cr, "advertising_id");
                } catch (Settings.SettingNotFoundException ex) {
                    // not supported
                    Log.w(TAG, "Error getting Amazon advertising info", ex);
                }
            } else {
                try {
                    idInfo = AdvertisingIdClient.getAdvertisingIdInfo(context);
                    if (idInfo != null) {
                        advertisingInfo.advertisingId = idInfo.getId();
                        advertisingInfo.limitAdTracking = idInfo.isLimitAdTrackingEnabled();
                    } else {
                        /// Advertising ID was not available through the Google Play Services. Panic!
                        //                    advertisingID = "PANIC!";
                    }
                } catch (NoClassDefFoundError ex) {
                    Log.e(TAG, "Play services Not available: " + ex.getLocalizedMessage());
                    ContentResolver cr = context.getContentResolver();
                    advertisingInfo.advertisingId = Settings.Secure.getString(cr, "advertising_id");
                } catch (GooglePlayServicesNotAvailableException exception) {
                    Log.e(TAG, "Play services Not available: " + exception.getLocalizedMessage());
                }
            }
        } catch (Exception ex) {
            Log.e(TAG, "Cannot load Advertising ID");
        }
        return advertisingInfo;
    }

    @Override
    public String getAppSetId() {
        //we try to get most recent value from cookie only if runtime value is not yet returned any result
        //this will happen only for very first request in session
        if (TextUtils.isEmpty(appSetId)) {
            Cookie appSetIdCookie = repository.load(APP_SET_ID_COOKIE, Cookie.class)
                    .get(timeoutProvider.getTimeout(), TimeUnit.MILLISECONDS);
            appSetId = (appSetIdCookie != null) ? appSetIdCookie.getString(APP_SET_ID) : null;
        }
        return appSetId;
    }

    @Override
    public String getAndroidId() {
        if (defaultIdFallbackDisabled) {
            return "";
        }
        @SuppressLint("HardwareIds")
        String androidId = Settings.Secure.getString(context.getContentResolver(), Settings.Secure.ANDROID_ID);
        return androidId;
    }

    @Override
    public void setAndroidIdFallbackDisabled(boolean disabled) {
        defaultIdFallbackDisabled = disabled;
    }

    @Override
    @Nullable
    public String getUserAgent() {
        String tempUserAgent;

        Cookie cookie = repository.load(USER_AGENT_ID_COOKIE, Cookie.class).get();
        if (cookie == null) {                            //Very first time or after clearing app data
            tempUserAgent = System.getProperty("http.agent");
        } else {
            tempUserAgent = cookie.getString("userAgent");
            if (TextUtils.isEmpty(tempUserAgent)) {        //Generally not possible but for safer side
                tempUserAgent = System.getProperty("http.agent");
            }
        }

        return tempUserAgent;
    }

    // AppSetIdInfo#getId might potentially return null causing a NPE. Possibly an issue with
    // Google's implementation.
    private void updateAppSetID() {
        //query value for AppSetId
        try {
            AppSetIdClient client = AppSet.getClient(context);
            Task<AppSetIdInfo> task = client.getAppSetIdInfo();
            task.addOnSuccessListener(new OnSuccessListener<AppSetIdInfo>() {
                @Override
                public void onSuccess(@NonNull AppSetIdInfo info) {
                    if (info != null) {
                        // Read app set ID value, which uses version 4 of the
                        // universally unique identifier (UUID) format.
                        appSetId = info.getId();
                        if (!TextUtils.isEmpty(appSetId)) {
                            Cookie appSetIdCookie = new Cookie(APP_SET_ID_COOKIE);
                            appSetIdCookie.putValue(APP_SET_ID, appSetId);
                            try {
                                repository.save(appSetIdCookie);
                            } catch (DatabaseHelper.DBException e) {
                                Log.e(TAG, "error saving AppSetId in Cookie: " + e.getLocalizedMessage());
                            }
                        }
                    }
                }
            });
        } catch (NoClassDefFoundError ex) {
            Log.e(TAG, "Required libs to get AppSetID Not available: " + ex.getLocalizedMessage());
        }
    }
}