package com.vungle.ads

import android.os.Build
import android.util.Log
import androidx.annotation.VisibleForTesting
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.executor.VungleThreadPoolExecutor
import com.vungle.ads.internal.util.ActivityManager
import com.vungle.ads.internal.util.Logger
import java.util.concurrent.*

/**
 * Main Class for tracking analytics
 */

object AnalyticsClient {

    enum class LogLevel(val level: Int) {
        ERROR_LOG_LEVEL_OFF(0), ERROR_LOG_LEVEL_ERROR(1), ERROR_LOG_LEVEL_DEBUG(2);

        companion object {
            fun fromValue(logLevel: Int): LogLevel {
                when (logLevel) {
                    ERROR_LOG_LEVEL_DEBUG.level -> {
                        return ERROR_LOG_LEVEL_DEBUG
                    }
                    ERROR_LOG_LEVEL_ERROR.level -> {
                        return ERROR_LOG_LEVEL_ERROR
                    }
                    ERROR_LOG_LEVEL_OFF.level -> {
                        return ERROR_LOG_LEVEL_OFF
                    }
                }
                return ERROR_LOG_LEVEL_ERROR
            }
        }
    }

    private val TAG: String = AnalyticsClient::class.java.simpleName
    @VisibleForTesting
    internal val errors: BlockingQueue<Sdk.SDKError.Builder> =
        LinkedBlockingQueue()
    @VisibleForTesting
    internal val metrics: BlockingQueue<Sdk.SDKMetric.Builder> =
        LinkedBlockingQueue()

    /**
     * Max time before any queued items are sent out
     */
    private const val refreshTimeMillis = 5000L

    /**
     * Max number of items queued before we send them out
     */
    private const val maxBatchSize = 20
    private var maxErrorLogLevel = Integer.MAX_VALUE

    @VisibleForTesting
    internal var vungleApiClient: VungleApiClient? = null

    @VisibleForTesting
    internal var executor: VungleThreadPoolExecutor? = null

    @VisibleForTesting
    internal var metricsEnabled = false

    private var logLevel = LogLevel.ERROR_LOG_LEVEL_ERROR
    private var paused = false

    internal fun init(
        vungleApiClient: VungleApiClient,
        executor: VungleThreadPoolExecutor,
        errorLogLevel: Int,
        metricsEnabled: Boolean,
    ) {

        this.executor = executor
        this.vungleApiClient = vungleApiClient
        this.metricsEnabled = metricsEnabled

        Executors.newSingleThreadScheduledExecutor()
            .scheduleAtFixedRate(
                {
                    report()
                },
                0,
                refreshTimeMillis,
                TimeUnit.MILLISECONDS
            )

        maxErrorLogLevel = errorLogLevel
        logLevel = LogLevel.fromValue(errorLogLevel)

        when (errorLogLevel) {
            LogLevel.ERROR_LOG_LEVEL_DEBUG.level -> {
                Logger.enable(true)
            }
            LogLevel.ERROR_LOG_LEVEL_ERROR.level -> {
                Logger.enable(false)
            }
            LogLevel.ERROR_LOG_LEVEL_OFF.level -> {
                Logger.enable(false)
            }
        }
        ActivityManager.instance.addListener(object : ActivityManager.LifeCycleCallback() {
            override fun onResume() {
                super.onResume()
                resume()
            }

            override fun onPause() {
                super.onPause()
                pause()
            }
        })
    }

    @Synchronized
    internal fun logError(
        reason: Sdk.SDKError.Reason,
        message: String,
        placementRefId: String? = null,
        creativeId: String? = null,
        eventId: String? = null,
    ) {
        if (logLevel == LogLevel.ERROR_LOG_LEVEL_OFF) {
            return
        }
        try {
            val error = Sdk.SDKError.newBuilder()
                .setOs("Android")
                .setOsVersion(Build.VERSION.SDK_INT.toString())
                .setMake(Build.MANUFACTURER)
                .setModel(Build.MODEL)
                .setReason(reason)
                .setMessage(message)
                .setAt(System.currentTimeMillis())
                .setPlacementReferenceId(placementRefId ?: "")
                .setCreativeId(creativeId ?: "")
                .setEventId(eventId ?: "")

            this.errors.put(error)
            if (BuildConfig.DEBUG) {
                Log.w(TAG, "Logging Error $reason with message \"$message\"")
            }
            if (errors.size >= maxBatchSize) {
                report()
            }
        } catch (ex: Exception) {
            Logger.e(TAG, "Cannot logError", ex)
        }

    }

    @Synchronized
    internal fun logError(
        reasonCode: Int,
        message: String,
        placementRefId: String? = null,
        creativeId: String? = null,
        eventId: String? = null,
    ) {
        logError(Sdk.SDKError.Reason.forNumber(reasonCode), message, placementRefId, creativeId, eventId)
    }

    /**
     * @param allowPending indicates whether logging of metric is allowed before init of AnalyticsClient
     */
    @Synchronized
    internal fun logMetric(
        metricType: Sdk.SDKMetric.SDKMetricType,
        metricValue: Long = 0,
        placementId: String? = null,
        creativeId: String? = null,
        eventId: String? = null,
        metaData: String? = null,
        allowPending: Boolean = false
    ) {
        if (!metricsEnabled && !allowPending) {
            return
        }
        if (BuildConfig.DEBUG) {
            Log.d(
                TAG, "Logging ${if (!metricsEnabled && allowPending) "Pending " else ""}" +
                        "Metric $metricType with value $metricValue for placement $placementId"
            )
        }
        val metric = Sdk.SDKMetric.newBuilder()
            .setType(metricType)
            .setValue(metricValue)
            .setMake(Build.MANUFACTURER)
            .setModel(Build.MODEL)
            .setOs("Android")
            .setOsVersion(Build.VERSION.SDK_INT.toString())

        metaData?.let {
            metric.meta = metaData
        }

        if (!placementId.isNullOrEmpty()) {
            metric.placementReferenceId = placementId
        }

        if (!creativeId.isNullOrEmpty()) {
            metric.creativeId = creativeId
        }

        if (!eventId.isNullOrEmpty()) {
            metric.eventId = eventId
        }

        this.metrics.put(metric)
        if (metrics.size >= maxBatchSize) {
            report()
        }
    }


    @Synchronized
    internal fun logMetric(
        metric: Metric,
        placementId: String? = null,
        creativeId: String? = null,
        eventId: String? = null,
        metaData: String? = null,
    ) {
        logMetric(
            metric.metricType,
            metric.getValue(),
            placementId,
            creativeId,
            eventId,
            metaData ?: metric.meta
        )
    }

    @Synchronized
    internal fun logMetric(
        singleValueMetric: SingleValueMetric,
        placementId: String? = null,
        creativeId: String? = null,
        eventId: String? = null,
        metaData: String? = null,
    ) {
        logMetric(
            singleValueMetric as Metric,
            placementId,
            creativeId,
            eventId,
            metaData
        )
    }

    @Synchronized
    internal fun logMetric(
        timeIntervalMetric: TimeIntervalMetric,
        placementId: String? = null,
        creativeId: String? = null,
        eventId: String? = null,
        metaData: String? = timeIntervalMetric.meta,
    ) {
        logMetric(
            timeIntervalMetric as Metric,
            placementId,
            creativeId,
            eventId,
            metaData
        )
    }

    @Synchronized
    internal fun logMetric(
        oneShotTimeIntervalMetric: OneShotTimeIntervalMetric,
        placementId: String? = null,
        creativeId: String? = null,
        eventId: String? = null,
        metaData: String? = null,
        ) {
        if (!oneShotTimeIntervalMetric.isLogged()) {
            logMetric(
                oneShotTimeIntervalMetric as TimeIntervalMetric,
                placementId,
                creativeId,
                eventId,
                metaData
            )
            oneShotTimeIntervalMetric.markLogged()
        }
    }

    @Synchronized
    private fun report() {
        if (paused) {
            return
        }

        if (logLevel != LogLevel.ERROR_LOG_LEVEL_OFF && errors.size > 0) {
            flushErrors()
        }

        if (metricsEnabled && metrics.size > 0) {
            flushMetrics()
        }
    }

    private fun flushMetrics() {
        Logger.d(TAG, message = "Sending ${this.metrics.size} metrics")
        executor?.execute {
            val currentSendingMetrics: BlockingQueue<Sdk.SDKMetric.Builder> =
                LinkedBlockingQueue()
            metrics.drainTo(currentSendingMetrics)
            if (currentSendingMetrics.isEmpty()) {/* There is a somewhat slim chance that upon getting
             here there is nothing to send. And we don't need those empty calls to server */
                return@execute
            }
            vungleApiClient?.reportMetrics(currentSendingMetrics, object : RequestListener {
                override fun onSuccess() {
                    Logger.d(TAG, message = "Sent ${currentSendingMetrics.size} metrics")
                }

                override fun onFailure() {
                    Logger.d(
                        TAG,
                        message = "Failed to send ${currentSendingMetrics.size} metrics"
                    )
                    metrics.addAll(currentSendingMetrics)
                }

            })
        }
    }

    private fun flushErrors() {
        Logger.d(TAG, message = "Sending ${this.errors.size} errors")
        executor?.execute {
            val currentSendingErrors: BlockingQueue<Sdk.SDKError.Builder> =
                LinkedBlockingQueue()
            errors.drainTo(currentSendingErrors)
            if (currentSendingErrors.isEmpty()) {/* There is a somewhat slim chance that upon getting
             here there is nothing to send. And we don't need those empty calls to server */
                return@execute
            }
            vungleApiClient?.reportErrors(currentSendingErrors, object : RequestListener {
                override fun onSuccess() {
                    Logger.d(TAG, message = "Sent ${currentSendingErrors.size} errors")
                }

                override fun onFailure() {
                    Logger.d(
                        TAG,
                        message = "Failed to send ${currentSendingErrors.size} errors"
                    )
                    errors.addAll(currentSendingErrors)
                }
            })
        }
    }

    fun pause() {
        paused = true
    }

    fun resume() {
        paused = false
    }

    interface RequestListener {
        fun onSuccess()
        fun onFailure()
    }

}
