package com.vungle.ads

import android.content.Context
import android.util.Log
import android.widget.RelativeLayout
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.executor.Executors
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.model.BidPayload
import com.vungle.ads.internal.model.Placement
import com.vungle.ads.internal.omsdk.OMTracker
import com.vungle.ads.internal.presenter.AdEventListener
import com.vungle.ads.internal.presenter.AdPlayCallback
import com.vungle.ads.internal.presenter.MRAIDPresenter
import com.vungle.ads.internal.ui.VungleWebClient
import com.vungle.ads.internal.ui.WatermarkView
import com.vungle.ads.internal.ui.view.MRAIDAdWidget
import com.vungle.ads.internal.util.ViewUtility.dpToPixels
import java.util.concurrent.atomic.AtomicBoolean

private const val TAG = "BannerView"

class BannerView(
    context: Context,
    placement: Placement,
    advertisement: AdPayload,
    adSize: BannerAdSize,
    adConfig: AdConfig,
    adPlayCallback: AdPlayCallback,
    bidPayload: BidPayload?
) : RelativeLayout(context) {

    private var calculatedPixelWidth = 0
    private var calculatedPixelHeight = 0

    private val adWidget: MRAIDAdWidget
    private val presenter: MRAIDPresenter
    private var imageView: WatermarkView? = null

    private var isOnImpressionCalled: Boolean = false

    private var destroyed = AtomicBoolean(false)

    override fun onWindowVisibilityChanged(visibility: Int) {
        super.onWindowVisibilityChanged(visibility)

        if (!isOnImpressionCalled) {
            return
        }

        // The visibility change must be triggered after impression fired.
        if (!destroyed.get()) {
            presenter.setAdVisibility(visibility == VISIBLE)
        }
    }

    fun onImpression() {
        isOnImpressionCalled = true
        presenter.start()
    }

    override fun onAttachedToWindow() {
        super.onAttachedToWindow()

        renderAd()
    }

    fun finishAdInternal(isFinishedByApi: Boolean) {
        if (destroyed.get()) {
            return
        }
        destroyed.set(true)
        val flag = (MRAIDAdWidget.AdStopReason.IS_AD_FINISHING
                or if (isFinishedByApi) MRAIDAdWidget.AdStopReason.IS_AD_FINISHED_BY_API else 0)
        presenter.stop()
        presenter.detach(flag)
        try {
            removeAllViews()
        } catch (e: Exception) {
            Log.d(TAG, "Removing webView error: $e")
        }
    }

    private fun renderAd() {
        if (visibility != VISIBLE) {
            return
        }

        if (adWidget.parent != this) {
            addView(adWidget, calculatedPixelWidth, calculatedPixelHeight)
            if (imageView != null) {
                addView(imageView, calculatedPixelWidth, calculatedPixelHeight)
                imageView?.bringToFront()
            }
        }

        val bannerLayoutParams = layoutParams
        if (bannerLayoutParams != null) {
            //Set Ad Size when Banner Ad is attached to Window
            bannerLayoutParams.height = calculatedPixelHeight
            bannerLayoutParams.width = calculatedPixelWidth
            requestLayout()
        }
    }

    init {
        //set Ad Size
        calculatedPixelHeight = dpToPixels(context, adSize.height)
        calculatedPixelWidth = dpToPixels(context, adSize.width)

        adWidget = MRAIDAdWidget(context)
        adWidget.setCloseDelegate(object : MRAIDAdWidget.CloseDelegate {
            override fun close() {
                finishAdInternal(false)
            }
        })

        val executors: Executors by inject(context)
        val omTrackerFactory: OMTracker.Factory by inject(context)
        val omTracker =
            omTrackerFactory.make(ConfigManager.omEnabled() && advertisement.omEnabled())
        val webClient = VungleWebClient(advertisement, placement, executors.offloadExecutor)

        webClient.setWebViewObserver(omTracker)
        presenter =
            MRAIDPresenter(
                adWidget,
                advertisement,
                placement,
                webClient,
                executors.jobExecutor,
                omTracker,
                bidPayload
            )
        presenter.setEventListener(object : AdEventListener(
            adPlayCallback,
            placement
        ) {})

        presenter.prepare()

        adConfig.getWatermark()?.let {
            imageView = WatermarkView(context, it)
        }

    }

}
