package com.vungle.ads.internal

import android.content.Context
import android.util.Log
import com.vungle.ads.*
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.VungleError.Companion.AD_EXPIRED
import com.vungle.ads.VungleError.Companion.AD_MARKUP_INVALID
import com.vungle.ads.VungleError.Companion.INVALID_AD_STATE
import com.vungle.ads.VungleError.Companion.VUNGLE_NOT_INITIALIZED
import com.vungle.ads.internal.downloader.Downloader
import com.vungle.ads.internal.executor.SDKExecutors
import com.vungle.ads.internal.load.*
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.model.BidPayload
import com.vungle.ads.internal.model.Placement
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.omsdk.OMInjector
import com.vungle.ads.internal.presenter.AdEventListener
import com.vungle.ads.internal.presenter.AdPlayCallback
import com.vungle.ads.internal.presenter.AdPlayCallbackWrapper
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.task.CleanupJob
import com.vungle.ads.internal.task.JobRunner
import com.vungle.ads.internal.ui.AdActivity
import com.vungle.ads.internal.util.ActivityManager
import com.vungle.ads.internal.util.PathProvider
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.json.Json

abstract class AdInternal(val context: Context) : AdLoaderCallback {
    companion object {
        private val TAG = AdInternal::class.simpleName

        private val THROW_ON_ILLEGAL_TRANSITION = BuildConfig.DEBUG

        private val json = Json {
            ignoreUnknownKeys = true
            encodeDefaults = true
            explicitNulls = false
        }
    }

    var adState: AdState = AdState.NEW
        set(value) {
            if (value.isTerminalState()) {
                advertisement?.eventId()?.let {
                    val jobRunner: JobRunner by inject(context)
                    val jobInfo = CleanupJob.makeJobInfo(it)
                    jobRunner.execute(jobInfo)
                }
            }

            field = adState.transitionTo(value)
        }

    var advertisement: AdPayload? = null
    var placement: Placement? = null
    var bidPayload: BidPayload? = null
    private var adLoaderCallback: AdLoaderCallback? = null
    private val vungleApiClient: VungleApiClient by inject(context)

    private var baseAdLoader: BaseAdLoader? = null

    private lateinit var requestMetric: TimeIntervalMetric

    fun canPlayAd(onPlay: Boolean = false): VungleError? {
        val error: VungleError =
            when {
                advertisement == null -> AdNotLoadedCantPlay()
                advertisement?.hasExpired() == true -> if (onPlay) {
                    AdExpiredOnPlayError()
                } else {
                    AdExpiredError()
                }

                adState == AdState.PLAYING -> ConcurrentPlaybackUnsupported()
                adState != AdState.READY -> InvalidAdState()
                else -> return null
            }
        if (onPlay) {
            error.setPlacementId(placement?.referenceId)
                .setCreativeId(advertisement?.getCreativeId())
                .setEventId(advertisement?.eventId())
                .logErrorNoReturnValue()
        }
        return error
    }

    abstract fun isValidAdTypeForPlacement(placement: Placement): Boolean

    abstract fun isValidAdSize(adSize: String): Boolean

    abstract fun getAdSizeForAdRequest(): String

    fun loadAd(
        placementId: String,
        adMarkup: String?,
        adLoaderCallback: AdLoaderCallback
    ) {
        this.adLoaderCallback = adLoaderCallback
        if (!VungleAds.isInitialized()) {
            adLoaderCallback.onFailure(InternalError(VUNGLE_NOT_INITIALIZED))
            return
        }

        val pl = ConfigManager.getPlacement(placementId)
        if (pl == null) {
            AnalyticsClient.logError(
                VungleError.INVALID_PLACEMENT_ID,
                "$placementId is invalid",
                placementId
            )
            adLoaderCallback.onFailure(InternalError(VungleError.PLACEMENT_NOT_FOUND))
            return
        }
        this.placement = pl
        if (!isValidAdTypeForPlacement(pl)) {
            adLoaderCallback.onFailure(InternalError(VungleError.TYPE_NOT_MATCH))
            return
        }
        val adSize = getAdSizeForAdRequest()
        if (!isValidAdSize(adSize)) {
            adLoaderCallback.onFailure(InternalError(VungleError.INVALID_SIZE))
            return
        }

        if (adState != AdState.NEW) {
            val error = when (adState) {
                AdState.NEW -> TODO() // Not possible
                AdState.LOADING -> VungleError.AD_IS_LOADING
                AdState.READY -> VungleError.AD_ALREADY_LOADED
                AdState.PLAYING -> VungleError.AD_IS_PLAYING
                AdState.FINISHED -> VungleError.AD_CONSUMED
                AdState.ERROR -> VungleError.AD_ALREADY_FAILED
            }
            AnalyticsClient.logError(
                error,
                "$adState state is incorrect for load",
                placementRefId = placementId,
                creativeId = advertisement?.getCreativeId(),
                eventId = advertisement?.eventId()
            )
            adLoaderCallback.onFailure(InternalError(INVALID_AD_STATE))
            return
        }

        val type =
            if (ConfigManager.adLoadOptimizationEnabled()) Sdk.SDKMetric.SDKMetricType.AD_REQUEST_TO_CALLBACK_ADO_DURATION_MS
            else Sdk.SDKMetric.SDKMetricType.AD_REQUEST_TO_CALLBACK_DURATION_MS
        requestMetric = TimeIntervalMetric(type)
        requestMetric.markStart()
        if (!adMarkup.isNullOrEmpty()) {
            try {
                bidPayload = json.decodeFromString<BidPayload>(adMarkup)
            } catch (e: IllegalArgumentException) {
                AnalyticsClient.logError(
                    VungleError.INVALID_ADUNIT_BID_PAYLOAD,
                    "Unable to decode payload into BidPayload object. Error: ${e.localizedMessage}",
                    placementId,
                    eventId = advertisement?.eventId(),
                    )
                adLoaderCallback.onFailure(InternalError(AD_MARKUP_INVALID))
                return
            } catch (e: Exception) {
                AnalyticsClient.logError(
                    VungleError.INVALID_JSON_BID_PAYLOAD,
                    "Unable to decode payload into BidPayload object. Error: ${e.localizedMessage}",
                    placementId,
                    eventId = advertisement?.eventId(),
                    )
                adLoaderCallback.onFailure(InternalError(AD_MARKUP_INVALID))
                return
            }
        }

        adState = AdState.LOADING

        val omInjector: OMInjector by inject(context)
        val sdkExecutors: SDKExecutors by inject(context)
        val pathProvider: PathProvider by inject(context)
        val downloader: Downloader by inject(context)

        if (adMarkup.isNullOrEmpty()) {
            val adLoader = DefaultAdLoader(
                context,
                vungleApiClient,
                sdkExecutors,
                omInjector,
                downloader,
                pathProvider
            )
            baseAdLoader = adLoader
            adLoader.loadAd(AdRequest(pl, null), adSize, this)
        } else {
            val adLoader = RealtimeAdLoader(
                context,
                vungleApiClient,
                sdkExecutors,
                omInjector,
                downloader,
                pathProvider
            )
            baseAdLoader = adLoader
            adLoader.loadAd(AdRequest(pl, bidPayload), adSize, this)
        }
    }

    internal fun cancelDownload() {
        baseAdLoader?.cancel()
    }

    fun play(adPlayCallback: AdPlayCallback) {
        val error = canPlayAd(true)
        if (error != null) {
            adPlayCallback.onFailure(error)
            if (isErrorTerminal(error.code)) {
                adState = AdState.ERROR
            }
            return
        }

        val pl = placement ?: return
        val adv = advertisement ?: return

        val callbackWrapper = object : AdPlayCallbackWrapper(adPlayCallback) {
            override fun onAdStart(id: String?) {
                adState = AdState.PLAYING
                super.onAdStart(id)
            }

            override fun onAdEnd(id: String?) {
                adState = AdState.FINISHED
                super.onAdEnd(id)
            }

            override fun onFailure(error: VungleError) {
                adState = AdState.ERROR
                super.onFailure(error)
            }
        }

        cancelDownload()

        renderAd(callbackWrapper, pl, adv)
    }

    internal open fun renderAd(
        listener: AdPlayCallback?,
        placement: Placement,
        advertisement: AdPayload
    ) {
        /// Subscribe to the event bus of the activity before starting the activity, otherwise
        /// the Publisher notices it has no subscribers and does not emit the start value.
        AdActivity.eventListener = object : AdEventListener(
            listener,
            placement
        ) {}

        AdActivity.advertisement = advertisement
        AdActivity.bidPayload = bidPayload

        /// Start the activity, and if there are any extras that have been overridden by the application, apply them.
        val intent = AdActivity.createIntent(context, placement.referenceId, advertisement.eventId())
        ActivityManager.startWhenForeground(context, null, intent, null)
    }

    override fun onSuccess(advertisement: AdPayload) {
        this@AdInternal.advertisement = advertisement
        adState = AdState.READY
        adLoadedAndUpdateConfigure(advertisement)
        adLoaderCallback?.onSuccess(advertisement)
        requestMetric.markEnd()
        AnalyticsClient.logMetric(requestMetric,
            placement?.referenceId, advertisement.getCreativeId(), advertisement.eventId())
    }

    internal open fun adLoadedAndUpdateConfigure(advertisement: AdPayload) {
    }

    override fun onFailure(error: VungleError) {
        adState = AdState.ERROR
        adLoaderCallback?.onFailure(error)
    }

    /*
      We use this method to check if we allowed to change ad state to ERROR. Because we use ERROR
    state to clean up the ad assets in the device.
      There's only one case we need to clean up assets which is pub try to play one READY but has
    expired ad.

    |InstanceState | canPlay() | expired | loadAd() | playAd() | playAd()-to-ERROR? |
    |--------------|-----------|---------|----------|----------|--------------------|
    |    NEW       |      10   |     -   |     Y    |    10    |            N       |
    |    LOADING   |      10   |     -   |     42   |    10    |            N       |
    |    READY     |      0    |     N   |     42   |     Y    |            N       |
    |    READY     |      4    |     Y   |     42   |     4    |            Y       |
    |    PLAYING   |    4,46   |    Y/N  |     42   |   4,46   |            N       |
    |    FINISHED  |    4,42   |    Y/N  |     42   |   4,42   |            N       |
    |    ERROR     |   10,4,42 |    Y/N  |     42   |  10,4,42 |            N       |

     */
    internal fun isErrorTerminal(errorCode: Int): Boolean {
        return adState == AdState.READY && errorCode == AD_EXPIRED
    }

    enum class AdState {
        NEW {
            override fun canTransitionTo(adState: AdState): Boolean {
                return adState === LOADING || adState === READY || adState === ERROR
            }
        },
        LOADING {
            override fun canTransitionTo(adState: AdState): Boolean {
                return adState === READY || adState === ERROR
            }
        },
        READY {
            override fun canTransitionTo(adState: AdState): Boolean {
                return adState === PLAYING || adState === ERROR
            }
        },
        PLAYING {
            override fun canTransitionTo(adState: AdState): Boolean {
                return adState === FINISHED || adState === ERROR
            }
        },
        FINISHED {
            override fun canTransitionTo(adState: AdState): Boolean {
                return false
            }
        },
        ERROR {
            override fun canTransitionTo(adState: AdState): Boolean {
                return adState === FINISHED
            }
        };

        abstract fun canTransitionTo(adState: AdState): Boolean

        fun transitionTo(adState: AdState): AdState {
            if (this != adState && !canTransitionTo(adState)) {
                val msg = "Cannot transition from ${this.name} to ${adState.name}"
                if (THROW_ON_ILLEGAL_TRANSITION) {
                    throw IllegalStateException(msg)
                } else {
                    Log.e(TAG, "Illegal state transition", IllegalStateException(msg))
                }
            }
            return adState
        }

        fun isTerminalState(): Boolean {
            return this in listOf(FINISHED, ERROR)
        }
    }
}
