package com.vungle.ads.internal.bidding

import android.os.Build
import android.util.Base64
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.VungleError
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.locale.LocaleInfo
import com.vungle.ads.internal.model.Cookie
import com.vungle.ads.internal.model.RtbTokens
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.persistence.FilePreferences
import com.vungle.ads.internal.platform.Platform
import com.vungle.ads.internal.privacy.PrivacyManager
import com.vungle.ads.internal.util.ActivityManager
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.SerializationException
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import java.io.ByteArrayOutputStream
import java.io.IOException
import java.util.zip.GZIPOutputStream

class BidTokenEncoder(
    private val localeInfo: LocaleInfo,
    private val platform: Platform,
    private val filePreferences: FilePreferences
) {

    private var ordinalView = 0
    private val json = Json { explicitNulls = false }

    private var enterBackgroundTime = 0L

    companion object {
        private const val TOKEN_VERSION = 3
    }

    init {
        ActivityManager.addLifecycleListener(object : ActivityManager.LifeCycleCallback() {
            override fun onResume() {
                super.onResume()
                if (System.currentTimeMillis() > enterBackgroundTime + ConfigManager.getSessionTimeoutInSecond() * 1000L) {
                    // Reset ordinal view count
                    ordinalView = 0
                    enterBackgroundTime = 0L
                }
            }

            override fun onPause() {
                super.onPause()
                enterBackgroundTime = System.currentTimeMillis()
            }
        })
    }

    fun encode(): String? {
        ordinalView++
        return bidTokenV3()
    }

    private fun bidTokenV3(
    ): String? {
        return try {
            val token: String = constructV3Token()
            val os = ByteArrayOutputStream(token.length)
            val gos = GZIPOutputStream(os)
            gos.write(token.toByteArray())
            gos.close()
            val compressed = os.toByteArray()
            val base64 = Base64.encodeToString(compressed, Base64.NO_WRAP)
            os.close()
            "$TOKEN_VERSION:$base64"
        } catch (e: IOException) {
            AnalyticsClient.logError(
                VungleError.GZIP_ENCODE_ERROR,
                "Fail to gzip bidtoken ${e.localizedMessage}"
            )
            null
        }
    }

    @Throws(SerializationException::class)
    private fun constructV3Token(): String {
        val ccpa = getCCPAStatus().let { RtbTokens.CCPA(it) }
        val gdpr = getGDPR()
        val coppa = getCOPPA()
        val consent = RtbTokens.Consent(ccpa, gdpr, coppa)

        val extension = RtbTokens.Extension(
            platform.isSideLoaded,
            platform.isSdCardPresent,
            platform.isSoundEnabled
        )

        var ifa: String? = null
        val isAmazonDevice = Platform.MANUFACTURER_AMAZON == Build.MANUFACTURER
        val androidInfo: AndroidInfo? = if (isAmazonDevice) null else AndroidInfo()
        val amazonInfo: AndroidInfo? = if (isAmazonDevice) AndroidInfo() else null
        if (PrivacyManager.shouldSendAdIds()) {
            val gaid = platform.getAdvertisingInfo()?.advertisingId
            // Android id will be retrieved if gaid is empty.
            val androidId = if (gaid.isNullOrEmpty()) platform.getAndroidId() else ""
            ifa = if (gaid.isNullOrEmpty()) androidId else gaid
            if (!androidId.isNullOrEmpty()) {
                if (isAmazonDevice) {
                    amazonInfo?.androidId = androidId
                } else {
                    androidInfo?.androidId = androidId
                }
            }
        }
        // App set id is always required.
        if (isAmazonDevice) {
            amazonInfo?.appSetId = platform.getAppSetId()
        } else {
            androidInfo?.appSetId = platform.getAppSetId()
        }

        val device = RtbTokens.Device(
            platform.isBatterySaverEnabled,
            localeInfo.timeZoneId,
            platform.volumeLevel,
            ifa,
            amazonInfo, androidInfo,
            localeInfo.language,
            extension
        )

        val request = RtbTokens.Request(
            getConfigExtension(),
            ordinalView,
            VungleApiClient.headerUa,
        )

        val tokens = RtbTokens(device, request, consent)
        return json.encodeToString(tokens)
    }

    private fun getCCPAStatus(): String {
        return PrivacyManager.getCcpaStatus()
    }

    private fun getGDPR(): RtbTokens.GDPR {
        return RtbTokens.GDPR(
            PrivacyManager.getConsentStatus(),
            PrivacyManager.getConsentSource(),
            PrivacyManager.getConsentMessageVersion(),
            PrivacyManager.getConsentTimestamp()
        )
    }

    private fun getCOPPA(): RtbTokens.COPPA {
        return RtbTokens.COPPA(PrivacyManager.getCoppaStatus().getValue())
    }

    private fun getConfigExtension(): String {
        //If config extension stored is not null send that value
        val configExtension = ConfigManager.getConfigExtension()
        if (configExtension.isEmpty()) {
            val storedConfigExtension = filePreferences.getString(Cookie.CONFIG_EXTENSION)
            storedConfigExtension?.let {
                return storedConfigExtension
            }
        }
        return configExtension
    }

    @Serializable
    data class AndroidInfo(
        @SerialName("android_id") var androidId: String? = null,
        @SerialName("app_set_id") var appSetId: String? = null
    )
}
