package com.vungle.ads.internal.load

import android.content.Context
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.InternalError
import com.vungle.ads.VungleError
import com.vungle.ads.VungleError.Companion.AD_FAILED_TO_DOWNLOAD
import com.vungle.ads.VungleError.Companion.NETWORK_ERROR
import com.vungle.ads.VungleError.Companion.NETWORK_TIMEOUT
import com.vungle.ads.VungleError.Companion.NO_SERVE
import com.vungle.ads.VungleError.Companion.SERVER_RETRY_ERROR
import com.vungle.ads.internal.downloader.Downloader
import com.vungle.ads.internal.executor.Executors
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.model.Placement
import com.vungle.ads.internal.network.Call
import com.vungle.ads.internal.network.Callback
import com.vungle.ads.internal.network.Response
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.omsdk.OMInjector
import com.vungle.ads.internal.util.PathProvider
import java.io.IOException
import java.net.SocketTimeoutException
import java.net.UnknownHostException

class DefaultAdLoader(
    context: Context,
    vungleApiClient: VungleApiClient,
    sdkExecutors: Executors,
    omInjector: OMInjector,
    downloader: Downloader,
    pathProvider: PathProvider
) : BaseAdLoader(
    context, vungleApiClient, sdkExecutors, omInjector, downloader, pathProvider
) {

    override fun requestAd() {
        fetchAdMetadata(adSize, adRequest.placement)
    }

    override fun onAdLoadReady() {
        // Do nothing.
    }

    private fun fetchAdMetadata(
        adSize: String,
        placement: Placement
    ) {
        if (vungleApiClient.checkIsRetryAfterActive()) {
            adLoaderCallback.onFailure(InternalError(SERVER_RETRY_ERROR))
            return
        }
        val adsCall = vungleApiClient.requestAd(
            placement.referenceId, adSize, placement.headerBidding
        )
        if (adsCall == null) {
            onAdLoadFailed(InternalError(AD_FAILED_TO_DOWNLOAD))
            return
        }
        adsCall.enqueue(object : Callback<AdPayload> {
            override fun onResponse(call: Call<AdPayload>?, response: Response<AdPayload>?) {

                sdkExecutors.backgroundExecutor.execute {
                    if (response?.isSuccessful == false) {
                        AnalyticsClient.logError(
                            VungleError.API_FAILED_STATUS_CODE,
                            "Failed to get a successful response from the API call",
                            placement.referenceId)
                        onAdLoadFailed(InternalError(NO_SERVE))
                    } else {
                        val adPayload = response?.body()
                        if (adPayload?.adUnit() == null) {
                            AnalyticsClient.logError(
                                VungleError.AD_RESPONSE_EMPTY,
                                "Ad markup is empty.",
                                placement.referenceId
                            )
                            onAdLoadFailed(
                                InternalError(NO_SERVE)
                            )
                            return@execute
                        }

                        try {
                            handleAdMetaData(adPayload)
                        } catch (ex: IllegalArgumentException) {

                            // TODO: is sleep logic still valid for new SDK?
                            /// Failed to parse the advertisement, which means there was no ad served.
                            /// Check for a sleep code and schedule a download job.
                            if (adPayload.adUnit()?.sleep != null) {
                                val sleep = adPayload.adUnit()?.sleep
                                AnalyticsClient.logError(
                                    VungleError.PLACEMENT_SLEEP, "Placement is sleep",
                                    placement.referenceId
                                )
                                /// Set this sleep time on the placement. We should not attempt to load
                                /// more ads for this placement ID until the time has elapsed.

                            }
                            onAdLoadFailed(InternalError(NO_SERVE))
                        }
                    }
                }
            }

            override fun onFailure(call: Call<AdPayload>?, t: Throwable?) {
                sdkExecutors.backgroundExecutor.execute {
                    val error: VungleError = retrofitToVungleError(t)
                    onAdLoadFailed(error)

                    when (error.code) {
                        NETWORK_TIMEOUT -> {
                            AnalyticsClient.logError(
                                VungleError.AD_RESPONSE_TIMED_OUT,
                                "Timeout for ads call.",
                                placement.referenceId,
                                advertisement?.getCreativeId(),
                                advertisement?.eventId(),
                            )
                        }
                        NETWORK_ERROR -> {
                            AnalyticsClient.logError(
                                VungleError.API_REQUEST_ERROR,
                                "Ads request error.",
                                placement.referenceId,
                                advertisement?.getCreativeId(),
                                advertisement?.eventId(),
                            )
                        }
                        else -> {
                            AnalyticsClient.logError(
                                VungleError.API_RESPONSE_DECODE_ERROR,
                                "Unable to decode ads response.",
                                placement.referenceId,
                                advertisement?.getCreativeId(),
                                advertisement?.eventId(),
                            )
                        }
                    }

                }
            }
        })

    }

    private fun retrofitToVungleError(throwable: Throwable?): VungleError {
        return when (throwable) {
            is UnknownHostException -> {
                InternalError(AD_FAILED_TO_DOWNLOAD)
            }
            is SocketTimeoutException -> {
                InternalError(NETWORK_TIMEOUT)
            }
            is IOException -> {
                InternalError(NETWORK_ERROR)
            }
            else -> {
                InternalError(AD_FAILED_TO_DOWNLOAD)
            }
        }
    }

}
