package com.vungle.ads.internal.model

import java.util.*

/**
 * A Placement is an object that represents a single ad configuration. Publishers use placements to
 * test ads in different locations within their app, or to try different kinds of campaigns for
 * A/B testing. The identifier is generated by the Vungle dashboard, and also contains a collection of
 * assets.
 */
class AdAsset internal constructor(
    adIdentifier: String, serverPath: String?, localPath: String,
    /**
     * Unique identifier for the Asset.
     */
    val identifier: String?
) {
    /**
     * Unique identifier for the Ad (Foreign key).
     */
    val adIdentifier: String?

    /**
     * Parent id, like ZIP for ZIP_ASSET
     */
    var parentId: String? = null

    /**
     * Server path of asset
     */
    val serverPath: String?

    /**
     * Local path of asset
     */
    val localPath: String?

    /**
     * Download Status
     */

    enum class Status {
        NEW, //for internal usage in AssetDownloader only
        DOWNLOAD_RUNNING,
        DOWNLOAD_FAILED,
        DOWNLOAD_SUCCESS,
        PROCESSED
    }

    var status: Status = Status.NEW

    /**
     * File type
     */
    enum class FileType {
        ZIP,
        ZIP_ASSET,
        ASSET,
    }
    lateinit var fileType: FileType

    /**
     * File size to check the integrity of file
     */
    var fileSize: Long

    /**
     * Retry Counter for each asset. //may be required in future
     */
    var retryCount: Int

    /**
     * Download Error Type
     */
    enum class ErrorType {
        NO_ERROR,
        CAN_RETRY_ERROR,
        CANNOT_RETRY_ERROR
    }

    var retryTypeError: ErrorType

    /**
     * Basic constructor, accepts an identifier for the placement.
     *
     * @param adIdentifier The foreign reference id of the placement.
     * @param serverPath   The server url.
     * @param localPath    The local file path.
     */
    constructor(adIdentifier: String, serverPath: String?, localPath: String) : this(
        adIdentifier,
        serverPath,
        localPath,
        UUID.randomUUID().toString()
    ) {
    }

    override fun equals(o: Any?): Boolean {
        if (this === o) return true
        if (o == null || javaClass != o.javaClass) return false
        val adAsset = o as AdAsset
        if (status != adAsset.status) return false
        if (fileType != adAsset.fileType) return false
        if (fileSize != adAsset.fileSize) return false
        if (retryCount != adAsset.retryCount) return false
        if (retryTypeError != adAsset.retryTypeError) return false
        if (if (identifier != null) identifier != adAsset.identifier else adAsset.identifier != null) return false
        if (if (adIdentifier != null) adIdentifier != adAsset.adIdentifier else adAsset.adIdentifier != null) return false
        if (if (parentId != null) parentId != adAsset.parentId else adAsset.parentId != null) return false
        if (if (serverPath != null) serverPath != adAsset.serverPath else adAsset.serverPath != null) return false
        return if (localPath != null) localPath == adAsset.localPath else adAsset.localPath == null
    }

    override fun hashCode(): Int {
        var result = identifier?.hashCode() ?: 0
        result = 31 * result + (adIdentifier?.hashCode() ?: 0)
        result = 31 * result + if (parentId != null) parentId.hashCode() else 0
        result = 31 * result + (serverPath?.hashCode() ?: 0)
        result = 31 * result + (localPath?.hashCode() ?: 0)
        result = 31 * result + status.hashCode()
        result = 31 * result + fileType.hashCode()
        result = 31 * result + (fileSize xor (fileSize ushr 32)).toInt()
        result = 31 * result + retryCount
        result = 31 * result + retryTypeError.hashCode()
        return result
    }

    override fun toString(): String {
        return "AdAsset{" +
                "identifier='" + identifier + '\'' +
                ", adIdentifier='" + adIdentifier + '\'' +
                ", serverPath='" + serverPath + '\'' +
                ", localPath='" + localPath + '\'' +
                ", status=" + status +
                ", fileType=" + fileType +
                ", fileSize=" + fileSize +
                ", retryCount=" + retryCount +
                ", retryTypeError=" + retryTypeError +
                '}'
    }

    init {
        this.adIdentifier = adIdentifier
        this.serverPath = serverPath
        this.localPath = localPath
        fileSize = -1
        retryCount = 0
        retryTypeError = ErrorType.NO_ERROR
    }
}