package com.vungle.ads.internal.downloader

import com.vungle.ads.AnalyticsClient
import com.vungle.ads.TimeIntervalMetric
import com.vungle.ads.internal.Constants
import com.vungle.ads.internal.model.AdAsset
import com.vungle.ads.internal.protos.Sdk
import java.util.concurrent.atomic.AtomicBoolean

class DownloadRequest(val priority: Priority, val asset: AdAsset, val placementId: String? = null,
                      val creativeId: String? = null, val eventId: String? = null) {

    private val cancelled = AtomicBoolean(false)

    private var downloadDuration: TimeIntervalMetric? = null

    val isTemplate: Boolean
        get() = asset.fileType == AdAsset.FileType.ZIP

    val isMainVideo: Boolean
        get() = asset.adIdentifier == Constants.KEY_MAIN_VIDEO

    override fun toString(): String {
        return "DownloadRequest{" +
                ", priority=" + priority +
                ", url='" + asset.serverPath + '\'' +
                ", path='" + asset.localPath + '\'' +
                ", cancelled=" + cancelled +
                ", placementId=" + placementId +
                ", creativeId=" + creativeId +
                ", eventId=" + eventId +
                '}'
    }

    enum class Priority(val priority: Int) {
        CRITICAL(-Int.MAX_VALUE), //for internal usage in AssetDownloader only
        HIGHEST(0),
        HIGH(1),
        LOWEST(Int.MAX_VALUE)

    }

    fun cancel() {
        cancelled.set(true)
    }

    fun isCancelled(): Boolean {
        return cancelled.get()
    }

    fun getPriority(): Int {
        return priority.priority
    }

    fun startRecord() {
        downloadDuration =
            TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.TEMPLATE_DOWNLOAD_DURATION_MS)
        downloadDuration?.markStart()
    }

    fun stopRecord() {
        downloadDuration?.let {
            it.markEnd()
            AnalyticsClient.logMetric(
                timeIntervalMetric = it,
                placementId = placementId,
                creativeId = creativeId,
                eventId = eventId,
                metaData = asset.serverPath
            )
        }
    }

}
