package com.vungle.ads.internal.load

import com.vungle.ads.internal.util.Logger
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.MraidJsError
import com.vungle.ads.VungleError
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.Constants
import com.vungle.ads.internal.downloader.AssetDownloadListener
import com.vungle.ads.internal.downloader.DownloadRequest
import com.vungle.ads.internal.downloader.Downloader
import com.vungle.ads.internal.executor.VungleThreadPoolExecutor
import com.vungle.ads.internal.model.AdAsset
import com.vungle.ads.internal.util.FileUtility
import com.vungle.ads.internal.util.PathProvider
import java.io.File

object MraidJsLoader {

    private const val TAG = "MraidJsLoader"

    const val MRAID_DOWNLOADED = 10
    const val MRAID_INVALID_ENDPOINT = 11
    const val MRAID_DOWNLOAD_FAILED = 12
    const val MRAID_AVAILABLE = 13

    fun downloadJs(
        pathProvider: PathProvider,
        downloader: Downloader,
        ioExecutor: VungleThreadPoolExecutor,
        onDownloadResult: (downloadState: Int) -> Unit
    ) {
        val mraidEndpoint = ConfigManager.getMraidEndpoint()
        if (mraidEndpoint.isNullOrEmpty()) {
            onDownloadResult(MRAID_INVALID_ENDPOINT)
            return
        }

        val mraidJsPath = pathProvider.getJsAssetDir(ConfigManager.getMraidJsVersion())
        val mraidJsFile = File(mraidJsPath, Constants.MRAID_JS_FILE_NAME)
        if (mraidJsFile.exists()) {
            onDownloadResult(MRAID_AVAILABLE)
            return
        }

        // delete invalid js first
        val jsPath = pathProvider.getJsDir()
        FileUtility.deleteContents(jsPath)

        // download latest js to temp file
        val tempFilePath = File("${mraidJsFile.absolutePath}-${System.currentTimeMillis()}.tmp")
        val mraidJsAsset = AdAsset(
            Constants.MRAID_JS_FILE_NAME,
            "$mraidEndpoint/${Constants.MRAID_JS_FILE_NAME}",
            tempFilePath.absolutePath,
            AdAsset.FileType.ASSET,
            true
        )
        val mraidDownloadRequest = DownloadRequest(DownloadRequest.Priority.HIGH, mraidJsAsset)
        downloader.download(mraidDownloadRequest, object : AssetDownloadListener {
            override fun onError(
                error: AssetDownloadListener.DownloadError?,
                downloadRequest: DownloadRequest
            ) {
                ioExecutor.execute {
                    val errorMessage = "download mraid js error: ${error?.serverCode}. " +
                            "Failed to load asset ${downloadRequest.asset.serverPath}"
                    Logger.d(TAG, errorMessage)
                    MraidJsError(errorMessage).logErrorNoReturnValue()

                    FileUtility.deleteContents(jsPath)
                    onDownloadResult(MRAID_DOWNLOAD_FAILED)
                }
            }

            override fun onSuccess(file: File, downloadRequest: DownloadRequest) {
                ioExecutor.execute {
                    if (mraidJsFile.exists()) {
                        Logger.w(TAG, "mraid js file already exists!")
                        onDownloadResult(MRAID_DOWNLOADED)
                        return@execute
                    }

                    if (file.exists() && file.length() > 0) {
                        file.copyTo(mraidJsFile, true)
                        FileUtility.deleteAndLogIfFailed(file)
                    }

                    if (mraidJsFile.exists() && mraidJsFile.length() > 0) {
                        onDownloadResult(MRAID_DOWNLOADED)
                    } else {
                        AnalyticsClient.logError(
                            VungleError.MRAID_JS_WRITE_FAILED,
                            "Mraid js downloaded but write failure: ${mraidJsFile.absolutePath}"
                        )

                        FileUtility.deleteContents(jsPath)
                        onDownloadResult(MRAID_DOWNLOAD_FAILED)
                    }
                }
            }
        })
    }

}
