package com.vungle.ads.internal.bidding

import android.content.Context
import androidx.annotation.VisibleForTesting
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.ServiceLocator
import com.vungle.ads.VungleError
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.model.RtbRequest
import com.vungle.ads.internal.model.RtbToken
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.util.ActivityManager
import com.vungle.ads.internal.util.InputOutputUtils
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import java.io.IOException

class BidTokenEncoder(
    private val context: Context
) {

    private var ordinalView = 0
    private val json = Json { explicitNulls = false }

    private var enterBackgroundTime = 0L

    companion object {
        private const val TOKEN_VERSION = 5
    }

    init {
        ActivityManager.addLifecycleListener(object : ActivityManager.LifeCycleCallback() {
            override fun onResume() {
                super.onResume()
                if (System.currentTimeMillis() > enterBackgroundTime + ConfigManager.getSessionTimeout()) {
                    // Reset ordinal view count
                    ordinalView = 0
                    enterBackgroundTime = 0L
                }
            }

            override fun onPause() {
                super.onPause()
                enterBackgroundTime = System.currentTimeMillis()
            }
        })
    }

    fun encode(): String? {
        ordinalView++
        return bidTokenV5()
    }

    private fun bidTokenV5():String? {
        return try {
            val token: String = constructV5Token()
            Logger.w("BidTokenEncoder", "BidToken: $token")
            val converted = InputOutputUtils.convertForSending(token)
            Logger.w("BidTokenEncoder", "After conversion: $TOKEN_VERSION:$converted")
            "$TOKEN_VERSION:$converted"
        } catch (e: Exception) {
            AnalyticsClient.logError(
                VungleError.GZIP_ENCODE_ERROR,
                "Fail to gzip bidtoken ${e.localizedMessage}"
            )
            null
        }
    }

    @VisibleForTesting
    internal fun constructV5Token(): String {
        val vungleApiClient: VungleApiClient by ServiceLocator.inject(context)
        val commonRequestBody = vungleApiClient.requestBody(!ConfigManager.signalsDisabled())
        val body = RtbToken(
            commonRequestBody.device,
            commonRequestBody.user,
            commonRequestBody.ext,
            RtbRequest(VungleApiClient.headerUa),
            ordinalView)
        return json.encodeToString(body)
    }

}
