package com.vungle.ads.fpd

import com.vungle.ads.internal.util.RangeUtil
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

@Serializable
class Demographic {

    @SerialName("email")
    private var email: String? = null

    @SerialName("phone_number")
    private var phoneNumber: String? = null

    @SerialName("age_range")
    private var ageRange: Int? = null

    @SerialName("yob")
    private var yob: Int? = null

    @SerialName("gender")
    private var gender: Int? = null

    @SerialName("education_level")
    private var educationLevel: Int? = null

    @SerialName("employment_status")
    private var employmentStatus: Int? = null

    @SerialName("locale_classification")
    private var localeClassification: Int? = null

    @SerialName("length_of_residence")
    private var lengthOfResidence: Int? = null

    @SerialName("median_home_value_usd")
    private var medianHomeValueUSD: Int? = null

    @SerialName("monthly_housing_payment_usd")
    private var monthlyHousingPaymentUSD: Int? = null

    @SerialName("ownership")
    private var ownership: Int? = null

    @SerialName("property_type")
    private var propertyType: Int? = null

    @SerialName("marital_status")
    private var maritalStatus: Int? = null

    @SerialName("income_usd")
    private var incomeUSD: Int? = null

    // Setter methods
    fun setEmail(email: String): Demographic {
        this.email = email
        return this
    }

    fun setPhoneNumber(phoneNumber: String): Demographic {
        this.phoneNumber = phoneNumber
        return this
    }

    fun setAgeRange(age: Int): Demographic {
        this.ageRange = AgeRange.fromAge(age).id
        return this
    }

    fun setYob(yob: Int): Demographic {
        if (RangeUtil.isInRange(yob, 1900)) {
            this.yob = yob
        }
        return this
    }

    fun setGender(gender: Gender): Demographic {
        this.gender = gender.id
        return this
    }

    fun setEducationLevel(educationLevel: EducationLevel): Demographic {
        this.educationLevel = educationLevel.id
        return this
    }

    fun setEmploymentStatus(employmentStatus: EmploymentStatus): Demographic {
        this.employmentStatus = employmentStatus.id
        return this
    }

    fun setLocaleClassification(localeClassification: LocationClassification): Demographic {
        this.localeClassification = localeClassification.id
        return this
    }

    fun setLengthOfResidence(lengthOfResidence: Int): Demographic {
        this.lengthOfResidence = LengthOfResidence.fromYears(lengthOfResidence).id
        return this
    }

    fun setMedianHomeValueUSD(homeValue: Int): Demographic {
        this.medianHomeValueUSD = MedianHomeValueUSD.fromPrice(homeValue).id
        return this
    }

    fun setMonthlyHousingCosts(housingCost: Int): Demographic {
        this.monthlyHousingPaymentUSD = MonthlyHousingCosts.fromCost(housingCost).id
        return this
    }

    fun setOwnershipStatus(ownershipStatus: OwnershipStatus): Demographic {
        this.ownership = ownershipStatus.id
        return this
    }

    fun setPropertyType(propertyType: PropertyType): Demographic {
        this.propertyType = propertyType.id
        return this
    }

    fun setMaritalStatus(maritalStatus: MaritalStatus): Demographic {
        this.maritalStatus = maritalStatus.id
        return this
    }

    fun setIncomeUSD(income: Int): Demographic {
        this.incomeUSD = IncomeBracket.fromIncome(income).id
        return this
    }
}

enum class Gender(val id: Int, val description: String) {
    UNKNOWN(0, "unknown"),
    MALE(1, "male"),
    FEMALE(2, "female")
}

// "description": "Education stages or highest level achieved.
// 0=Other, 1=No Schooling, 2=Some High School, 3=High School Graduate, 4=Some College,
// 5=Bachelor's Degree, 6=Master's Degree, 7=Professional Degree, 8=Doctorate",
enum class EducationLevel(val id: Int, val description: String) {
    OTHER(0, "Other"),
    NO_SCHOOLING(1, "No Schooling"),
    SOME_HIGH_SCHOOL(2, "Some High School"),
    HIGH_SCHOOL_GRADUATE(3, "High School Graduate"),
    SOME_COLLEGE(4, "Some College"),
    BACHELORS_DEGREE(5, "Bachelor's Degree"),
    MASTERS_DEGREE(6, "Master's Degree"),
    PROFESSIONAL_DEGREE(7, "Professional Degree"),
    DOCTORATE(8, "Doctorate")
}


enum class AgeRange(val id: Int, val range: IntRange) {
    AGE_18_20(1, 18..20),
    AGE_21_30(2, 21..30),
    AGE_31_40(3, 31..40),
    AGE_41_50(4, 41..50),
    AGE_51_60(5, 51..60),
    AGE_61_70(6, 61..70),
    AGE_71_75(7, 71..75),
    OTHERS(0, Int.MIN_VALUE..Int.MAX_VALUE);  // Default range for others

    companion object {
        internal fun fromAge(age: Int): AgeRange {
            return values().firstOrNull { age in it.range } ?: OTHERS
        }
    }
}

enum class IncomeBracket(val id: Int, val range: IntRange) {
    UNDER_10K(0, Int.MIN_VALUE..10000),
    FROM_10K_TO_30K(1, 10001..30000),
    FROM_30K_TO_50K(2, 30001..50000),
    FROM_50K_TO_70K(3, 50001..70000),
    FROM_70K_TO_100K(4, 70001..100000),
    FROM_100K_TO_200K(5, 100001..200000),
    FROM_200K_TO_300K(6, 200001..300000),
    FROM_300K_TO_500K(7, 300001..500000),
    FROM_500K_TO_700K(8, 500001..700000),
    OVER_700K(9, 700001..Int.MAX_VALUE);

    companion object {
        internal fun fromIncome(income: Int): IncomeBracket {
            return values().firstOrNull { income in it.range } ?: UNDER_10K
        }
    }
}

enum class MaritalStatus(val id: Int, val description: String) {
    SINGLE(0, "Single"),
    MARRIED(1, "Married"),
    DIVORCED(2, "Divorced"),
    WIDOWED(3, "Widowed"),
    SEPARATED(4, "Separated"),
    PARTNERED(5, "Partnered"),
    ENGAGED(6, "Engaged"),
    COHABITING(7, "Co-habiting"),
    ANNULLED(8, "Annulled"),
    COMMON_LAW(9, "Common Law")
}

enum class PropertyType(val id: Int, val description: String) {
    OTHER(0, "Other"),
    SINGLE_FAMILY_HOME(1, "Single Family Home"),
    MULTI_FAMILY_UNIT(2, "Multi-Family Unit"),
    CONDO(3, "Condo"),
    TOWNHOUSE(4, "Townhouse"),
    MOBILE_HOME(5, "Mobile Home"),
    APARTMENT(6, "Apartment"),
    STUDIO(7, "Studio"),
    FARMHOUSE(8, "Farmhouse"),
    RANCH(9, "Ranch")
}

enum class OwnershipStatus(val id: Int, val description: String) {
    OTHER(0, "Other"),
    OWNER_OCCUPIED(1, "Owner Occupied"),
    RENTER_OCCUPIED(2, "Renter Occupied"),
    LEASED(3, "Leased"),
    SHARED(4, "Shared"),
    FREEHOLD(5, "Freehold"),
    CO_OP(6, "Co-op"),
    STATE_OWNED(7, "State Owned"),
    MANAGED(8, "Managed"),
    VACATION_HOME(9, "Vacation Home")
}

enum class MonthlyHousingCosts(val id: Int, val range: IntRange) {
    UNDER_500(0, Int.MIN_VALUE..500),
    FROM_500_TO_1000(1, 501..1000),
    FROM_1000_TO_1500(2, 1001..1500),
    FROM_1500_TO_2000(3, 1501..2000),
    FROM_2000_TO_2500(4, 2001..2500),
    FROM_2500_TO_3000(5, 2501..3000),
    FROM_3000_TO_3500(6, 3001..3500),
    FROM_3500_TO_4000(7, 3501..4000),
    FROM_4000_TO_4500(8, 4001..4500),
    OVER_4500(9, 4501..Int.MAX_VALUE);

    companion object {
        internal fun fromCost(cost: Int): MonthlyHousingCosts {
            return values().firstOrNull { cost in it.range } ?: UNDER_500
        }
    }
}

enum class MedianHomeValueUSD(val id: Int, val range: IntRange) {
    UNDER_100K(0, Int.MIN_VALUE..100000),
    FROM_100K_TO_300K(1, 100001..300000),
    FROM_300K_TO_500K(2, 300001..500000),
    FROM_500K_TO_700K(3, 500001..700000),
    FROM_700K_TO_900K(4, 700001..900000),
    FROM_900K_TO_1M1(5, 900001..1100000),
    FROM_1M1_TO_1M3(6, 1100001..1300000),
    FROM_1M3_TO_1M5(7, 1300001..1500000),
    FROM_1M5_TO_1M7(8, 1500001..1700000),
    OVER_1M7(9, 1700001..Int.MAX_VALUE);

    companion object {
        internal fun fromPrice(price: Int): MedianHomeValueUSD {
            return values().firstOrNull { price in it.range } ?: UNDER_100K
        }
    }
}

//"description": "Duration of residence at the current location.
// 0=<1 year, 1=1-5 years, 2=6-10 years, 3=11-20 years, 4=21-30 years,
// 5=31-40 years, 6=41-50 years, 7=51-60 years, 8=61-70 years, 9=71-75+ years",
enum class LengthOfResidence(val id: Int, val range: IntRange) {
    LESS_THAN_ONE_YEAR(0, Int.MIN_VALUE..0),
    ONE_TO_FIVE_YEARS(1, 1..5),
    SIX_TO_TEN_YEARS(2, 6..10),
    ELEVEN_TO_TWENTY_YEARS(3, 11..20),
    TWENTY_ONE_TO_THIRTY_YEARS(4, 21..30),
    THIRTY_ONE_TO_FORTY_YEARS(5, 31..40),
    FORTY_ONE_TO_FIFTY_YEARS(6, 41..50),
    FIFTY_ONE_TO_SIXTY_YEARS(7, 51..60),
    SIXTY_ONE_TO_SEVENTY_YEARS(8, 61..70),
    OVER_SEVENTY_ONE_YEARS(9, 71..Int.MAX_VALUE);

    companion object {
        internal fun fromYears(years: Int): LengthOfResidence {
            return values().firstOrNull { years in it.range } ?: LESS_THAN_ONE_YEAR
        }
    }
}

//"description": "General geographic area of residence.
// 0=Other, 1=Urban, 2=Suburban, 3=Rural, 4=Town, 5=City Center,
// 6=Residential Area, 7=Outskirts, 8=Industrial Area, 9=Village",
enum class LocationClassification(val id: Int, val description: String) {
    OTHER(0, "Other"),
    URBAN(1, "Urban"),
    SUBURBAN(2, "Suburban"),
    RURAL(3, "Rural"),
    TOWN(4, "Town"),
    CITY_CENTER(5, "City Center"),
    RESIDENTIAL_AREA(6, "Residential Area"),
    OUTSKIRTS(7, "Outskirts"),
    INDUSTRIAL_AREA(8, "Industrial Area"),
    VILLAGE(9, "Village")
}

// "description": "Current employment status indicating economic activity.
// 0=Other, 1=Employed, 2=Unemployed, 3=Self-Employed, 4=Retired, 5=Student, 6=Intern, 7=Temporary, 8=Freelancer, 9=Casual",
enum class EmploymentStatus(val id: Int, val description: String) {
    OTHER(0, "Other"),
    EMPLOYED(1, "Employed"),
    UNEMPLOYED(2, "Unemployed"),
    SELF_EMPLOYED(3, "Self-Employed"),
    RETIRED(4, "Retired"),
    STUDENT(5, "Student"),
    INTERN(6, "Intern"),
    TEMPORARY(7, "Temporary"),
    FREELANCER(8, "Freelancer"),
    CASUAL(9, "Casual")
}
