package com.vungle.ads.internal

import android.content.Context
import android.net.Uri
import androidx.annotation.VisibleForTesting
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.ConfigurationError
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.VungleError
import com.vungle.ads.internal.executor.SDKExecutors
import com.vungle.ads.internal.model.ConfigExtension
import com.vungle.ads.internal.model.ConfigPayload
import com.vungle.ads.internal.model.Cookie
import com.vungle.ads.internal.model.Placement
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.omsdk.OMInjector
import com.vungle.ads.internal.persistence.FilePreferences
import com.vungle.ads.internal.privacy.PrivacyManager
import com.vungle.ads.internal.util.Logger
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json

object ConfigManager {
    const val TAG = "ConfigManager"
    private const val DEFAULT_SESSION_TIMEOUT_SECONDS = 15 * 60
    private const val DEFAULT_SIGNALS_SESSION_TIMEOUT_SECONDS = 30 * 60
    private var config: ConfigPayload? = null
    private var endpoints: ConfigPayload.Endpoints? = null
    private var placements: List<Placement>? = null
    private var configExt: String? = null
    private val json = Json {
        ignoreUnknownKeys = true
        encodeDefaults = true
        explicitNulls = false
    }

    internal fun fetchConfig(context: Context): ConfigPayload? {
        val vungleApiClient: VungleApiClient by inject(context)
        try {
            val response = vungleApiClient.config()?.execute()
            if (response == null || !response.isSuccessful) {
                return null
            }

            val configPayload = response.body() ?: return null

            if (configPayload.endpoints == null || !validateEndpoints(configPayload.endpoints)
                || configPayload.placements == null
            ) {
                return null
            }

            return configPayload
        } catch (throwable: Throwable) {
            Logger.e(TAG, "Error while fetching config: ${throwable.message}")
            return null
        }
    }

    @VisibleForTesting
    internal fun updateConfigExtension(context: Context, ext: String) {
        this.configExt = ext
        val filePreferences: FilePreferences by inject(context)
        filePreferences.put(Cookie.CONFIG_EXTENSION, ext).apply()
    }

    private fun updateConfigSwitchThread(context: Context) {
        val sdkExecutors: SDKExecutors by inject(context)
        sdkExecutors.backgroundExecutor.execute {
            val newConfigPayload = fetchConfig(context)
            if (newConfigPayload != null) {
                initWithConfig(context, newConfigPayload, false)
            } else {
                ConfigurationError().logErrorNoReturnValue()
            }
        }
    }

    internal fun onConfigExtensionReceived(context: Context, ext: ConfigExtension) {
        //if configExt is non-null, cache it anyway
        ext.configExt?.let { extString ->
            val sdkExecutors: SDKExecutors by inject(context)
            sdkExecutors.backgroundExecutor.execute {
                updateConfigExtension(context, extString)
            }
        }
        //if needRefresh flag is set, do the full update
        if (ext.needRefresh == true) {
            updateConfigSwitchThread(context)
        }
    }

    fun initWithConfig(
        context: Context,
        config: ConfigPayload,
        fromCachedConfig: Boolean,
        appId: String? = null
    ) {
        this.config = config
        endpoints = config.endpoints
        placements = config.placements

        val filePreferences: FilePreferences by inject(context)

        // Update config payload
        if (!fromCachedConfig) {
            updateCachedConfig(config, filePreferences, appId)
        }

        // Update config extension
        config.configExtension?.let { updateConfigExtension(context, it) }

        // Init OM SDK
        if (omEnabled()) {
            val omInjector: OMInjector by inject(context)
            omInjector.init()
        }

        // Update privacy
        PrivacyManager.updateDisableAdId(shouldDisableAdId())
    }

    fun placements() = placements

    fun getPlacement(id: String): Placement? {
        return placements?.find {
            it.referenceId == id
        }
    }

    fun getAdsEndpoint(): String? {
        return endpoints?.adsEndpoint
    }

    fun getRiEndpoint(): String? {
        return endpoints?.riEndpoint
    }

    fun getMraidEndpoint(): String? {
        return endpoints?.mraidEndpoint
    }

    fun getMraidJsVersion(): String {
        return getMraidEndpoint()?.let {
            "mraid_${Uri.parse(it).lastPathSegment}"
        } ?: "mraid_1"
    }

    fun getGDPRConsentMessage(): String? {
        return config?.userPrivacy?.gdpr?.consentMessage
    }

    fun getGDPRConsentTitle(): String? {
        return config?.userPrivacy?.gdpr?.consentTitle
    }

    fun getGDPRButtonAccept(): String? {
        return config?.userPrivacy?.gdpr?.buttonAccept
    }

    fun getGDPRButtonDeny(): String? {
        return config?.userPrivacy?.gdpr?.buttonDeny
    }

    fun getGDPRConsentMessageVersion(): String {
        return config?.userPrivacy?.gdpr?.consentMessageVersion ?: ""
    }

    fun getGDPRIsCountryDataProtected(): Boolean {
        return config?.userPrivacy?.gdpr?.isCountryDataProtected ?: false
    }

    fun shouldDisableAdId(): Boolean {
        return config?.disableAdId ?: true
    }

    fun isReportIncentivizedEnabled(): Boolean {
        return config?.isReportIncentivizedEnabled ?: false
    }

    fun getConfigExtension(): String {
        return configExt ?: ""
    }

    fun omEnabled() = config?.viewAbility?.om ?: false

    fun getMetricsEndpoint(): String? {
        return endpoints?.metricsEndpoint
    }

    fun getErrorLoggingEndpoint(): String? {
        return endpoints?.errorLogsEndpoint
    }

    fun getMetricsEnabled(): Boolean {
        return config?.logMetricsSettings?.metricsEnabled ?: false
    }

    fun getLogLevel(): Int {
        return config?.logMetricsSettings?.errorLogLevel
            ?: AnalyticsClient.LogLevel.ERROR_LOG_LEVEL_ERROR.level
    }

    fun getSessionTimeout(): Long {
        return (config?.sessionTimeout ?: DEFAULT_SESSION_TIMEOUT_SECONDS) * 1000L
    }

    fun getSignalsSessionTimeout(): Long {
        return (config?.signalSessionTimeout ?: DEFAULT_SIGNALS_SESSION_TIMEOUT_SECONDS) * 1000L
    }

    fun rtaDebuggingEnabled(): Boolean {
        return config?.rtaDebugging ?: false
    }

    fun isCacheableAssetsRequired(): Boolean {
        return config?.isCacheableAssetsRequired ?: false
    }

    fun signalsDisabled(): Boolean {
        return config?.signalsDisabled ?: false
    }

    fun fpdEnabled(): Boolean {
        return config?.fpdEnabled ?: true
    }

    @VisibleForTesting
    internal fun validateEndpoints(endpoints: ConfigPayload.Endpoints? = this.endpoints): Boolean {
        var valid = true
        if (endpoints?.adsEndpoint.isNullOrEmpty()) {
            AnalyticsClient.logError(
                VungleError.INVALID_ADS_ENDPOINT,
                "The ads endpoint was not provided in the config."
            )
            valid = false
        }
        if (endpoints?.riEndpoint.isNullOrEmpty()) {
            AnalyticsClient.logError(
                VungleError.INVALID_RI_ENDPOINT,
                "The ri endpoint was not provided in the config."
            )
        }
        if (endpoints?.mraidEndpoint.isNullOrEmpty()) {
            AnalyticsClient.logError(
                VungleError.MRAID_DOWNLOAD_JS_ERROR,
                "The mraid endpoint was not provided in the config."
            )
            valid = false
        }
        if (endpoints?.metricsEndpoint.isNullOrEmpty()) {
            AnalyticsClient.logError(
                VungleError.INVALID_METRICS_ENDPOINT,
                "The metrics endpoint was not provided in the config."
            )
        }
        if (endpoints?.errorLogsEndpoint.isNullOrEmpty()) {
            /* uncomment this if there is a fallback errorLogsEndpoint or logged errors are stored
            to send when valid endpoint is provided
            AnalyticsClient.logError(
                VungleError.INVALID_LOG_ERROR_ENDPOINT,
                "The error logging endpoint was not provided in the config."
            )
            */
            Logger.e(TAG, "The error logging endpoint was not provided in the config.")
        }
        return valid
    }

    fun isCleverCacheEnabled(): Boolean {
        return config?.cleverCache?.enabled ?: false
    }

    fun getCleverCacheDiskSize(): Long {
        return config?.cleverCache?.diskSize?.also { return it * 1024 * 1024 }
            ?: (1000 * 1024 * 1024)
    }

    fun getCleverCacheDiskPercentage(): Int {
        return config?.cleverCache?.diskPercentage ?: 3
    }

    fun getCachedConfig(filePreferences: FilePreferences, appId: String): ConfigPayload? {
        try {
            val localAppId = filePreferences.getString(Cookie.CONFIG_APP_ID)
            if (localAppId.isNullOrEmpty() || !localAppId.equals(appId, ignoreCase = true)) {
                Logger.w(TAG, "app id mismatch, re-config")
                return null
            }

            filePreferences.getString(Cookie.CONFIG_RESPONSE)?.let {
                val configLastUpdateTime = filePreferences.getLong(Cookie.CONFIG_UPDATE_TIME, 0)
                val localConfig = json.decodeFromString<ConfigPayload>(it)
                if ((localConfig.configSettings?.refreshTime
                        ?: -1) + configLastUpdateTime < System.currentTimeMillis()
                ) {
                    Logger.w(TAG, "cache config expired. re-config")
                    return null
                }
                Logger.w(TAG, "use cache config.")

                return localConfig
            }
        } catch (e: Exception) {
            Logger.e(TAG, "Error while parsing cached config: ${e.message}")
            return null
        }
        return null
    }

    fun updateCachedConfig(
        config: ConfigPayload,
        filePreferences: FilePreferences,
        appId: String? = null,
    ) {
        try {
            appId?.let { filePreferences.put(Cookie.CONFIG_APP_ID, appId) }
            filePreferences.put(Cookie.CONFIG_UPDATE_TIME, System.currentTimeMillis())
            filePreferences.put(Cookie.CONFIG_RESPONSE, json.encodeToString(config))
            filePreferences.apply()
        } catch (e: Exception) {
            Logger.e(TAG, "Exception: ${e.message} for updating cached config")
        }
    }

}
