package com.vungle.ads.internal

import android.Manifest
import android.content.Context
import androidx.annotation.VisibleForTesting
import androidx.core.content.PermissionChecker
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.ConfigurationError
import com.vungle.ads.InitializationListener
import com.vungle.ads.InvalidAppId
import com.vungle.ads.NetworkPermissionsNotGranted
import com.vungle.ads.NetworkUnreachable
import com.vungle.ads.OutOfMemory
import com.vungle.ads.SdkAlreadyInitialized
import com.vungle.ads.SdkInitializationInProgress
import com.vungle.ads.SdkVersionTooLow
import com.vungle.ads.ServiceLocator
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.TimeIntervalMetric
import com.vungle.ads.UnknownConfigurationError
import com.vungle.ads.VungleError
import com.vungle.ads.internal.downloader.Downloader
import com.vungle.ads.internal.executor.Executors
import com.vungle.ads.internal.load.MraidJsLoader
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.persistence.FilePreferences
import com.vungle.ads.internal.platform.Platform
import com.vungle.ads.internal.privacy.PrivacyManager
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.signals.SignalManager
import com.vungle.ads.internal.task.CleanupJob
import com.vungle.ads.internal.task.JobRunner
import com.vungle.ads.internal.task.ResendTpatJob
import com.vungle.ads.internal.util.ActivityManager
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.PathProvider
import com.vungle.ads.internal.util.ThreadUtil
import java.net.UnknownHostException
import java.util.concurrent.atomic.AtomicBoolean

internal class VungleInitializer {

    companion object {
        private const val TAG = "VungleInitializer"
    }

    @VisibleForTesting
    internal var isInitialized = false

    @VisibleForTesting
    internal var isInitializing = AtomicBoolean(false)
    private var initRequestToResponseMetric: TimeIntervalMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.INIT_REQUEST_TO_RESPONSE_DURATION_MS)

    fun init(
        appId: String,
        context: Context,
        initializationCallback: InitializationListener
    ) {

        ActivityManager.init(context)

        if (isAppIdInvalid(appId)) {
            onInitError(
                initializationCallback,
                InvalidAppId().logError()
            )
            return
        }

        val platform: Platform by inject(context)

        if (!platform.isAtLeastMinimumSDK) {
            Logger.e(TAG, "SDK is supported only for API versions 21 and above")
            onInitError(
                initializationCallback,
                SdkVersionTooLow().logError()
            )
            return
        }

        if (isInitialized()) {
            Logger.d(TAG, "init already complete")
            SdkAlreadyInitialized().logErrorNoReturnValue()
            onInitSuccess(initializationCallback)
            return
        }

        if (isInitializing.getAndSet(true)) {
            Logger.d(TAG, "init ongoing")
            onInitError(
                initializationCallback,
                SdkInitializationInProgress().logError()
            )
            return
        }

        if (PermissionChecker.checkCallingOrSelfPermission(
                context,
                Manifest.permission.ACCESS_NETWORK_STATE
            )
            != PermissionChecker.PERMISSION_GRANTED
            || PermissionChecker.checkCallingOrSelfPermission(context, Manifest.permission.INTERNET)
            != PermissionChecker.PERMISSION_GRANTED
        ) {
            Logger.e(TAG, "Network permissions not granted")
            onInitError(
                initializationCallback,
                NetworkPermissionsNotGranted()
            )
            return
        }

        val sdkExecutors: Executors by inject(context)
        val vungleApiClient: VungleApiClient by inject(context)
        sdkExecutors.backgroundExecutor.execute(
            {
                PrivacyManager.init(context)

                vungleApiClient.initialize(appId)

                /// Configure the instance.
                configure(context, appId, initializationCallback)
            }
        ) {
            //failRunnable
            onInitError(
                initializationCallback,
                OutOfMemory("Config: Out of Memory").logError()
            )
        }
    }

    private fun isAppIdInvalid(appId: String) = appId.isBlank()

    /**
     * Request a configuration from the server. This updates the current list of placements and
     * schedules another configuration job in the future. It also has the side-effect of loading the
     * auto_cached ad if it is not downloaded currently.
     *
     * @param callback Callback that will be called when initialization has completed or failed.
     */
    private fun configure(context: Context, appId: String, callback: InitializationListener) {
        val vungleApiClient: VungleApiClient by inject(context)
        /// Request a configuration from the server. This happens asynchronously on the network thread.
        try {
            var fromCachedConfig = true
            val filePreferences: FilePreferences by inject(context)
            var initialConfigPayload = ConfigManager.getCachedConfig(filePreferences, appId)

            if (initialConfigPayload == null) {
                initRequestToResponseMetric.markStart()
                initialConfigPayload = ConfigManager.fetchConfig(context)
                initRequestToResponseMetric.markEnd()

                fromCachedConfig = false
            }

            if (initialConfigPayload == null) {
                onInitError(callback, ConfigurationError().logError())
                return
            }

            ConfigManager.initWithConfig(context, initialConfigPayload, fromCachedConfig, appId)

            // Init metric & error logging.
            val sdkExecutors: Executors by inject(context)
            val signalManager: SignalManager by inject(context)
            AnalyticsClient.init(
                vungleApiClient,
                sdkExecutors.loggerExecutor,
                ConfigManager.getLogLevel(),
                ConfigManager.getMetricsEnabled(),
                signalManager
            )

            val jobRunner: JobRunner by inject(context)
            jobRunner.execute(CleanupJob.makeJobInfo())
            jobRunner.execute(ResendTpatJob.makeJobInfo())

            isInitialized = true
            // Inform the publisher that initialization has succeeded.
            onInitSuccess(callback)

            // Download js assets asynchronously
            val pathProvider: PathProvider by inject(context)
            val downloader: Downloader by inject(context)
            MraidJsLoader.downloadJs(pathProvider, downloader, sdkExecutors.backgroundExecutor)
        } catch (throwable: Throwable) {
            isInitialized = false
            Logger.e(TAG, "Cannot finish init", throwable)
            when (throwable) {
                is UnknownHostException, is SecurityException -> {
                    onInitError(callback, NetworkUnreachable().logError())

                }

                is VungleError -> {
                    onInitError(callback, throwable)
                }

                else -> {
                    onInitError(callback, UnknownConfigurationError().logError())
                }
            }
        }
    }

    private fun onInitError(initCallback: InitializationListener, exception: VungleError) {
        isInitializing.set(false)
        ThreadUtil.runOnUiThread {
            initCallback.onError(exception)
        }
        val exMsg = exception.localizedMessage ?: "Exception code is ${exception.code}"
        Logger.e(TAG, exMsg)
    }

    private fun onInitSuccess(initCallback: InitializationListener) {
        isInitializing.set(false)
        ThreadUtil.runOnUiThread {
            Logger.d(TAG, "onSuccess")
            initCallback.onSuccess()
        }
        AnalyticsClient.logMetric(
            metric = initRequestToResponseMetric,
            metaData = VungleApiClient.BASE_URL
        )
    }

    fun isInitialized(): Boolean {
        return isInitialized
    }

    /**
     * Lite version of de-initialization, can be used in Unit Tests
     */
    internal fun deInit() {
        ServiceLocator.deInit()
        VungleApiClient.reset()
        isInitialized = false
        isInitializing.set(false)
    }

}
