package com.vungle.ads.internal.bidding

import android.content.Context
import androidx.annotation.VisibleForTesting
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.ServiceLocator
import com.vungle.ads.VungleError
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.model.RtbRequest
import com.vungle.ads.internal.model.RtbToken
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.util.ActivityManager
import com.vungle.ads.internal.util.InputOutputUtils
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json

private const val TAG = "BidTokenEncoder"

class BidTokenEncoder(
    private val context: Context
) {

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal var ordinalView = 0

    @OptIn(ExperimentalSerializationApi::class)
    private val json = Json {
        ignoreUnknownKeys = true
        encodeDefaults = true
        explicitNulls = false
    }

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal var enterBackgroundTime = 0L

    companion object {
        internal const val TOKEN_VERSION = 6
    }

    init {
        ActivityManager.addLifecycleListener(object : ActivityManager.LifeCycleCallback() {
            override fun onResume() {
                super.onResume()
                this@BidTokenEncoder.onResume()
            }

            override fun onPause() {
                super.onPause()
                this@BidTokenEncoder.onPause()
            }
        })
    }


    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal fun onResume() {
        /* the case for first call of onResume, that has no onPause before it.
        Also fixes case with multiple onResume */
        if (enterBackgroundTime == 0L) {
            Logger.d(TAG, "BidTokenEncoder#onResume skipped")
            return
        }

        val currentTimeMillis = System.currentTimeMillis()
        val sessionTimeout = ConfigManager.getSessionTimeout()
        if (currentTimeMillis > enterBackgroundTime + sessionTimeout) {
            // Reset ordinal view count
            ordinalView = 0
            enterBackgroundTime = 0L
        }
    }

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal fun onPause() {
        enterBackgroundTime = System.currentTimeMillis()
    }

    fun encode(): String? {
        ordinalView++
        return generateBidToken()
    }

    private fun generateBidToken():String? {
        return try {
            val token: String = constructV6Token()
            Logger.d(TAG, "BidToken: $token")
            val converted = InputOutputUtils.convertForSending(token)
            Logger.d(TAG, "After conversion: $TOKEN_VERSION:$converted")
            "$TOKEN_VERSION:$converted"
        } catch (e: Exception) {
            AnalyticsClient.logError(
                VungleError.GZIP_ENCODE_ERROR,
                "Fail to gzip bidtoken ${e.localizedMessage}"
            )
            null
        }
    }

    @VisibleForTesting
    internal fun constructV6Token(): String {
        val vungleApiClient: VungleApiClient by ServiceLocator.inject(context)
        val commonRequestBody = vungleApiClient.requestBody(!ConfigManager.signalsDisabled(), ConfigManager.fpdEnabled())
        val body = RtbToken(
            commonRequestBody.device,
            commonRequestBody.user,
            commonRequestBody.ext,
            RtbRequest(VungleApiClient.headerUa),
            ordinalView)
        return json.encodeToString(body)
    }

}
