package com.vungle.ads.internal.network

import com.vungle.ads.internal.util.Logger
import androidx.annotation.VisibleForTesting
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.TpatRetryFailure
import com.vungle.ads.internal.Constants
import com.vungle.ads.internal.persistence.FilePreferences
import com.vungle.ads.internal.persistence.FilePreferences.Companion.TPAT_FAILED_FILENAME
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.signals.SignalManager
import com.vungle.ads.internal.util.PathProvider
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import java.util.concurrent.Executor
import java.util.regex.Pattern

class TpatSender(
    val vungleApiClient: VungleApiClient,
    val placementId: String?,
    val creativeId: String?,
    val eventId: String?,
    ioExecutor: Executor,
    pathProvider: PathProvider,
    val signalManager: SignalManager? = null
) {
    companion object {
        private const val TAG: String = "TpatSender"
        private const val FAILED_TPATS = "FAILED_TPATS"
        private const val MAX_RETRIES = 5
    }

    private val tpatFilePreferences = FilePreferences.get(ioExecutor, pathProvider, TPAT_FAILED_FILENAME)

    fun sendWinNotification(urlString: String, executor: Executor) {
        val url = injectSessionIdToUrl(urlString)
        executor.execute {
            val error = vungleApiClient.pingTPAT(url)
            if (error != null) {
                AnalyticsClient.logError(
                    Sdk.SDKError.Reason.AD_WIN_NOTIFICATION_ERROR,
                    "Fail to send $url, error: ${error.description}",
                    placementId, creativeId, eventId
                )
            }
        }
    }

    fun sendTpats(urls: Iterable<String>, executor: Executor) {
        urls.forEach { sendTpat(it, executor) }
    }

    fun sendTpat(url: String, executor: Executor) {
        val urlWithSessionId = injectSessionIdToUrl(url)
        executor.execute {
            val storedTpats = getStoredTpats()
            val attemptNumber = storedTpats[url] ?: 0
            val error = vungleApiClient.pingTPAT(urlWithSessionId)
            if (error == null) {
                if (attemptNumber != 0) {
                    storedTpats.remove(urlWithSessionId)
                    saveStoredTpats(storedTpats)
                }
            } else {
                if (!error.errorIsTerminal) {
                    if (attemptNumber >= MAX_RETRIES) {
                        storedTpats.remove(url)
                        saveStoredTpats(storedTpats)
                        TpatRetryFailure(urlWithSessionId).logErrorNoReturnValue()
                    } else {
                        storedTpats[url] = attemptNumber + 1
                        saveStoredTpats(storedTpats)
                    }
                }
                Logger.e(TAG, "TPAT failed with ${error.description}, url:$urlWithSessionId")
                if (error.reason == Sdk.SDKMetric.SDKMetricType.NOTIFICATION_REDIRECT_VALUE) {
                    AnalyticsClient.logMetric(
                        metricType = Sdk.SDKMetric.SDKMetricType.NOTIFICATION_REDIRECT,
                        placementId = placementId,
                        metaData = urlWithSessionId
                    )
                } else {
                    AnalyticsClient.logError(
                        Sdk.SDKError.Reason.TPAT_ERROR,
                        "Fail to send $urlWithSessionId, error: ${error.description}",
                        placementId, creativeId, eventId
                    )
                }
            }
        }
    }

    private fun getStoredTpats(): HashMap<String, Int> {
        val storedTpats = tpatFilePreferences.getString(FAILED_TPATS)
        return try {
            if (storedTpats != null) {
                Json.decodeFromString(storedTpats)
            } else {
                HashMap()
            }
        } catch (e: Exception) {
            Logger.e(TAG, "Failed to decode stored tpats: $storedTpats")
            HashMap()
        }
    }

    private fun saveStoredTpats(tpats: HashMap<String, Int>) {
        try {
            tpatFilePreferences
                .put(FAILED_TPATS, Json.encodeToString(tpats))
                .apply()
        } catch (e: Exception) {
            Logger.e(TAG, "Failed to encode the about to storing tpats: $tpats")
        }
    }

    internal fun resendStoredTpats(executor: Executor) {
        getStoredTpats().forEach { (url, _) -> sendTpat(url, executor) }
    }

    @VisibleForTesting
    fun injectSessionIdToUrl(url: String): String {
        val sessionId = signalManager?.uuid ?: ""
        return if (sessionId.isNotEmpty()) {
            url.replace(
                Pattern.quote(Constants.SESSION_ID).toRegex(), sessionId
            )
        } else {
            url
        }
    }
}
