package com.vungle.ads.internal.util

import android.os.Build
import android.webkit.URLUtil
import androidx.annotation.VisibleForTesting
import com.vungle.ads.BuildConfig
import com.vungle.ads.internal.util.FileUtility.ObjectInputStreamProvider
import okhttp3.HttpUrl.Companion.toHttpUrlOrNull
import java.io.Closeable
import java.io.File
import java.io.FileInputStream
import java.io.FileOutputStream
import java.io.IOException
import java.io.InputStream
import java.io.ObjectInputStream
import java.io.ObjectOutputStream
import java.io.Serializable
import java.nio.file.Files


internal object FileUtility {
    @get:VisibleForTesting
    @set:VisibleForTesting
    var objectInputStreamProvider: ObjectInputStreamProvider =
        ObjectInputStreamProvider { inputStream ->
            SafeObjectInputStream(inputStream, allowedClasses)
        }
    private val TAG = FileUtility::class.java.simpleName

    @VisibleForTesting
    internal val allowedClasses = listOf<Class<*>>(
        LinkedHashSet::class.java,
        HashSet::class.java,
        HashMap::class.java,
        ArrayList::class.java,
        File::class.java
    )

    @JvmStatic
    fun printDirectoryTree(folder: File?) {
        if (!BuildConfig.DEBUG) return
        if (folder == null) {
            Logger.d(TAG, "File is null ")
            return
        }
        if (!folder.exists()) {
            Logger.d(TAG, "File does not exist " + folder.path)
            return
        }
        if (!folder.isDirectory) {
            Logger.d(TAG, "File is not a directory " + folder.path)
            return
        }
        val indent = 0
        val sb = StringBuilder()
        printDirectoryTree(folder, indent, sb)
        Logger.d("Vungle", sb.toString())
    }

    private fun printDirectoryTree(folder: File?, indent: Int, sb: StringBuilder) {
        if (folder == null) {
            return
        }
        require(folder.isDirectory) { "folder is not a Directory" }
        sb.append(getIndentString(indent)).append("+--").append(folder.name).append("/\n")
        val files = folder.listFiles() ?: return
        for (file in files) {
            if (file.isDirectory) {
                printDirectoryTree(file, indent + 1, sb)
            } else {
                printFile(file, indent + 1, sb)
            }
        }
    }

    private fun printFile(file: File, indent: Int, sb: StringBuilder) {
        sb.append(getIndentString(indent)).append("+--").append(file.name).append('\n')
    }

    private fun getIndentString(indent: Int): String {
        val sb = StringBuilder()
        for (i in 0 until indent) {
            sb.append("|  ")
        }
        return sb.toString()
    }

    /**
     * Helper method to recursively delete a directory. Used to clean up assets by their identifying
     * folder names.
     *
     * @param f The file or directory to delete.
     * @throws IOException if deleting a file fails for any reason.
     */
    @JvmStatic
    fun delete(f: File?) {
        try {
            if (f == null || !f.exists()) return
            if (f.isDirectory) {
                deleteContents(f)
            }
            if (!f.delete()) {
                Logger.d(TAG, "Failed to delete file: $f")
            }
        } catch (e: Exception) {
            Logger.e(TAG, "Failed to delete file: ${e.localizedMessage}")
        }
    }

    @JvmStatic
    fun deleteContents(folder: File) {
        val filesInFolder = folder.listFiles() ?: return
        for (fileInFolder in filesInFolder) {
            delete(fileInFolder)
        }
    }

    @JvmStatic
    fun deleteAndLogIfFailed(file: File) {
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                Files.delete(file.toPath())
            } else if (!file.delete()) {
                Logger.e(TAG, "Cannot delete " + file.name)
            }
        } catch (e: Exception) {
            Logger.e(TAG, "Cannot delete " + file.name, e)
        }
    }

    fun closeQuietly(closeable: Closeable?) {
        try {
            closeable?.close()
        } catch (ignored: IOException) {
            //do nothing
        }
    }

    @JvmStatic
    fun writeSerializable(
        file: File,
        serializable: Serializable?
    ) {
        if (file.exists()) deleteAndLogIfFailed(file)
        if (serializable == null) return
        var fout: FileOutputStream? = null
        var oout: ObjectOutputStream? = null
        try {
            fout = FileOutputStream(file)
            oout = ObjectOutputStream(fout)
            oout.writeObject(serializable)
            oout.reset()
        } catch (e: IOException) {
            Logger.e(TAG, "${e.message}")
        } finally {
            closeQuietly(oout)
            closeQuietly(fout)
        }
    }

    fun writeString(file: File, content: String?) {
        if (content == null) return
        try {
            file.writeText(content, Charsets.UTF_8)
        } catch (e: IOException) {
            Logger.e(TAG, "${e.message}")
        }
    }

    @JvmStatic
    fun <T> readSerializable(file: File): T? {
        if (!file.exists()) return null
        var fin: FileInputStream? = null
        var oin: ObjectInputStream? = null
        try {
            fin = FileInputStream(file)
            oin = objectInputStreamProvider.provideObjectInputStream(fin)
            return oin.readObject() as T
        } catch (e: IOException) {
            Logger.e(TAG, "IOException: ${e.message}")
        } catch (e: ClassNotFoundException) {
            Logger.e(TAG, "ClassNotFoundException: ${e.message}")
        } catch (e: Exception) {
            //TODO after implementing logging send this corrupted file to us for investigation
            Logger.e(TAG, "cannot read serializable ${e.message}")
        } finally {
            closeQuietly(oin)
            closeQuietly(fin)
        }
        try {
            delete(file)
        } catch (ignored: IOException) {
            //ignored
        }
        return null
    }

    fun readString(file: File): String? {
        if (!file.exists()) return null
        try {
            return file.readText()
        } catch (e: IOException) {
            Logger.e(TAG, "IOException: ${e.message}")
        } catch (e: Exception) {
            Logger.e(TAG, "cannot read string ${e.message}")
        }
        return null
    }

    fun isValidUrl(httpUrl: String?): Boolean {
        return !httpUrl.isNullOrEmpty() && httpUrl.toHttpUrlOrNull() != null
    }

    fun size(file: File?): Long {
        if (file == null || !file.exists()) return 0
        var length: Long = 0
        if (file.isDirectory) {
            val children = file.listFiles()
            if (children != null && children.isNotEmpty()) {
                for (child in children) {
                    length += size(child)
                }
            }
            return length
        }
        return file.length()
    }

    fun guessFileName(url: String, ext: String? = null): String {
        return URLUtil.guessFileName(url, null, ext)
    }

    internal fun interface ObjectInputStreamProvider {
        @Throws(IOException::class, ClassNotFoundException::class)
        fun provideObjectInputStream(inputStream: InputStream?): ObjectInputStream
    }
}
