package com.vungle.ads.internal.model

/**
 * A Placement is an object that represents a single ad configuration. Publishers use placements to
 * test ads in different locations within their app, or to try different kinds of campaigns for
 * A/B testing. The identifier is generated by the Vungle dashboard, and also contains a collection of
 * assets.
 *
 * adIdentifier: Ad asset TOKEN in cacheable placements or template.
 * serverPath: Server path of asset
 * localPath: Local path of asset
 * fileType: Type of asset
 * isRequired: If true, then this asset is required to load before playing the ad.
 */
class AdAsset(
    val adIdentifier: String, val serverPath: String, val localPath: String,
    val fileType: FileType, val isRequired: Boolean
) {

    /**
     * Download Status
     */
    enum class Status {
        NEW, //for internal usage in AssetDownloader only
        DOWNLOAD_RUNNING,
        DOWNLOAD_FAILED,
        DOWNLOAD_SUCCESS,
        PROCESSED
    }

    /**
     * File type
     */
    enum class FileType {
        ZIP,
        ASSET,
    }

    var status: Status = Status.NEW

    /**
     * File size to check the integrity of file
     */
    var fileSize: Long = 0

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other == null || javaClass != other.javaClass) return false
        val adAsset = other as AdAsset
        if (status != adAsset.status) return false
        if (fileType != adAsset.fileType) return false
        if (fileSize != adAsset.fileSize) return false
        if (isRequired != adAsset.isRequired) return false
        if (adIdentifier != adAsset.adIdentifier) return false
        if (serverPath != adAsset.serverPath) return false
        return localPath == adAsset.localPath
    }

    override fun hashCode(): Int {
        var result = adIdentifier.hashCode()
        result = 31 * result + serverPath.hashCode()
        result = 31 * result + localPath.hashCode()
        result = 31 * result + status.hashCode()
        result = 31 * result + fileType.hashCode()
        result = 31 * result + (fileSize xor (fileSize ushr 32)).toInt()
        result = 31 * result + isRequired.hashCode()
        return result
    }

    override fun toString(): String {
        return "AdAsset{" +
                ", adIdentifier='" + adIdentifier + '\'' +
                ", serverPath='" + serverPath + '\'' +
                ", localPath='" + localPath + '\'' +
                ", status=" + status +
                ", fileType=" + fileType +
                ", fileSize=" + fileSize +
                ", isRequired=" + isRequired +
                '}'
    }

}
