package com.vungle.ads

import android.content.Context
import com.vungle.ads.internal.AdInternal
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.presenter.AdPlayCallback
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.util.ThreadUtil

abstract class BaseFullscreenAd(context: Context, placementId: String, adConfig: AdConfig) :
    BaseAd(context, placementId, adConfig), FullscreenAd {

    override fun load(adMarkup: String?) {
        signaledAd = signalManager.getSignaledAd(placementId)
        super.load(adMarkup)
    }

    override fun onAdLoaded(advertisement: AdPayload) {
        super.onAdLoaded(advertisement)
        signaledAd?.adAvailabilityCallbackTime = System.currentTimeMillis()
    }

    override fun play(context: Context?) {
        AnalyticsClient.logMetric(
            SingleValueMetric(Sdk.SDKMetric.SDKMetricType.PLAY_AD_API),
            placementId, creativeId, eventId
        )
        responseToShowMetric.markEnd()
        AnalyticsClient.logMetric(responseToShowMetric, placementId, creativeId, eventId)
        showToPresentMetric.markStart()
        showToFailMetric.markStart()
        signaledAd?.let {
            it.playAdTime = System.currentTimeMillis()
            it.calculateTimeBetweenAdAvailabilityAndPlayAd()
            signalManager.registerSignaledAd(context, it)
        }
        adInternal.play(context, object : AdPlayCallback {
            override fun onAdStart(id: String?) {
                signalManager.increaseSessionDepthCounter()
                showToPresentMetric.markEnd()
                AnalyticsClient.logMetric(showToPresentMetric, placementId, creativeId, eventId)
                presentToDisplayMetric.markStart()
                ThreadUtil.runOnUiThread {
                    adListener?.onAdStart(this@BaseFullscreenAd)
                }
            }

            override fun onAdImpression(id: String?) {
                ThreadUtil.runOnUiThread {
                    adListener?.onAdImpression(this@BaseFullscreenAd)
                }

                presentToDisplayMetric.markEnd()
                AnalyticsClient.logMetric(presentToDisplayMetric, placementId, creativeId, eventId)
                displayToClickMetric.markStart()
            }

            override fun onAdEnd(id: String?) {
                ThreadUtil.runOnUiThread {
                    adListener?.onAdEnd(this@BaseFullscreenAd)
                }
            }

            override fun onAdClick(id: String?) {
                ThreadUtil.runOnUiThread {
                    adListener?.onAdClicked(this@BaseFullscreenAd)
                }

                displayToClickMetric.markEnd()
                AnalyticsClient.logMetric(displayToClickMetric, placementId, creativeId, eventId)
            }

            override fun onAdRewarded(id: String?) {
                ThreadUtil.runOnUiThread {
                    val rewardedAdListener: RewardedAdListener? =
                        adListener as? RewardedAdListener
                    rewardedAdListener?.onAdRewarded(this@BaseFullscreenAd)
                }
            }

            override fun onAdLeftApplication(id: String?) {
                ThreadUtil.runOnUiThread {
                    adListener?.onAdLeftApplication(this@BaseFullscreenAd)
                }
            }

            override fun onFailure(error: VungleError) {
                ThreadUtil.runOnUiThread {
                    adListener?.onAdFailedToPlay(this@BaseFullscreenAd, error)
                }
                showToFailMetric.markEnd()
                AnalyticsClient.logMetric(showToFailMetric, placementId, creativeId, eventId)
            }
        })
    }
}

internal abstract class FullscreenAdInternal(context: Context) : AdInternal(context) {
    override fun isValidAdSize(adSize: VungleAdSize?): Boolean {
        return true // Don't care about the adSize for fullscreen ad type
    }

    override fun getAdSizeForAdRequest(): VungleAdSize? {
        return null
    }
}
