package com.vungle.ads.internal

import android.content.Context
import android.net.Uri
import androidx.annotation.VisibleForTesting
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.ConfigurationError
import com.vungle.ads.NetworkUnreachable
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.SingleValueMetric
import com.vungle.ads.TimeIntervalMetric
import com.vungle.ads.VungleError
import com.vungle.ads.internal.Constants.DEFAULT_ADS_ENDPOINT
import com.vungle.ads.internal.Constants.DEFAULT_ERROR_LOGS_ENDPOINT
import com.vungle.ads.internal.Constants.DEFAULT_METRICS_ENDPOINT
import com.vungle.ads.internal.model.ConfigPayload
import com.vungle.ads.internal.model.Cookie
import com.vungle.ads.internal.model.Placement
import com.vungle.ads.internal.network.Call
import com.vungle.ads.internal.network.Callback
import com.vungle.ads.internal.network.Response
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.omsdk.OMInjector
import com.vungle.ads.internal.persistence.FilePreferences
import com.vungle.ads.internal.privacy.PrivacyManager
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.util.Logger
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import java.net.UnknownHostException


object ConfigManager {
    const val TAG = "ConfigManager"
    const val CONFIG_LAST_VALIDATE_TS_DEFAULT = -1L

    private const val DEFAULT_SESSION_TIMEOUT_SECONDS = 15 * 60
    private const val DEFAULT_SIGNALS_SESSION_TIMEOUT_SECONDS = 30 * 60

    private const val CONFIG_NOT_AVAILABLE = 0
    private const val CONFIG_LAST_VALIDATED_TIMESTAMP_ONLY = 1
    private const val CONFIG_ALL_DATA = 2

    private var config: ConfigPayload? = null
    private var endpoints: ConfigPayload.Endpoints? = null
    private var placements: List<Placement>? = null
    private var configExt: String? = null
    private val json = Json {
        ignoreUnknownKeys = true
        encodeDefaults = true
        explicitNulls = false
    }

    private lateinit var applicationId: String

    internal fun setAppId(applicationId: String) {
        this.applicationId = applicationId
    }

    internal fun fetchConfigAsync(
        context: Context,
        onComplete: (result: Boolean) -> Unit
    ) {
        val vungleApiClient: VungleApiClient by inject(context)
        try {
            val initRequestToResponseMetric =
                TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.INIT_REQUEST_TO_RESPONSE_DURATION_MS)
            initRequestToResponseMetric.markStart()
            vungleApiClient.config()?.enqueue(object : Callback<ConfigPayload> {
                override fun onResponse(
                    call: Call<ConfigPayload>?,
                    response: Response<ConfigPayload>?
                ) {
                    initRequestToResponseMetric.markEnd()
                    AnalyticsClient.logMetric(
                        metric = initRequestToResponseMetric,
                        metaData = VungleApiClient.BASE_URL
                    )
                    if (response == null || !response.isSuccessful || response.body() == null) {
                        ConfigurationError().logErrorNoReturnValue()
                        return
                    }
                    val configPayload = response.body()
                    val configSrcMetric = SingleValueMetric(Sdk.SDKMetric.SDKMetricType.CONFIG_LOADED_FROM_INIT)

                    // apply local cache.
                    initWithConfig(context, configPayload, false, configSrcMetric)
                    onComplete(true)
                }

                override fun onFailure(call: Call<ConfigPayload>?, t: Throwable?) {
                    initRequestToResponseMetric.markEnd()
                    AnalyticsClient.logMetric(
                        metric = initRequestToResponseMetric,
                        metaData = VungleApiClient.BASE_URL
                    )
                    ConfigurationError().logErrorNoReturnValue()
                    Logger.e(TAG, "Error while fetching config: ${t?.message}")
                    onComplete(false)
                }

            })
        } catch (throwable: Throwable) {
            when (throwable) {
                is UnknownHostException, is SecurityException -> {
                    NetworkUnreachable().logErrorNoReturnValue()
                }
                else -> {
                    ConfigurationError().logErrorNoReturnValue()
                }
            }
            onComplete(false)
        }
    }

    @VisibleForTesting
    internal fun updateConfigExtension(context: Context, ext: String) {
        this.configExt = ext
        val filePreferences: FilePreferences by inject(context)
        filePreferences.put(Cookie.CONFIG_EXTENSION, ext).apply()
    }

    @Synchronized
    internal fun initWithConfig(
        context: Context,
        config: ConfigPayload?,
        fromCachedConfig: Boolean,
        metric: SingleValueMetric? = null
    ) {
        try {
            val filePreferences: FilePreferences by inject(context)
            val configStatus = checkConfigPayload(config)
            when (configStatus) {
                CONFIG_NOT_AVAILABLE -> {
                    Logger.e(TAG, "Config is not available.")
                    return
                }
                CONFIG_LAST_VALIDATED_TIMESTAMP_ONLY -> {
                    if (!fromCachedConfig && config != null) {
                        val timestamp = config.configLastValidatedTimestamp ?: CONFIG_LAST_VALIDATE_TS_DEFAULT
                        this.config?.configLastValidatedTimestamp = timestamp
                        // Update config cache even only configLastValidatedTimestamp is changed.
                        this.config?.let { updateCachedConfig(it, filePreferences) }
                    }
                    return
                }
                else -> {
                    this.config = config
                    endpoints = config?.endpoints
                    placements = config?.placements
                    AnalyticsClient.updateErrorLevelAndMetricEnabled(
                        getLogLevel(),
                        getMetricsEnabled())
                    // Update config payload
                    if (!fromCachedConfig && config != null) {
                        updateCachedConfig(config, filePreferences)
                        // Update config extension
                        config.configExtension?.let { updateConfigExtension(context, it) }
                    }

                    // Init OM SDK
                    if (omEnabled()) {
                        val omInjector: OMInjector by inject(context)
                        omInjector.init()
                    }
                    metric?.let { AnalyticsClient.logMetric(metric) }

                    // Update privacy
                    PrivacyManager.updateDisableAdId(shouldDisableAdId())

                }
            }
        } catch (e: Exception) {
            Logger.e(TAG, "Error while validating config: ${e.message}")
            return
        }
    }

    fun placements() = placements

    fun getPlacement(id: String): Placement? {
        return placements?.find {
            it.referenceId == id
        }
    }

    fun getAdsEndpoint(): String {
        return endpoints?.adsEndpoint.takeUnless { it.isNullOrEmpty() } ?: DEFAULT_ADS_ENDPOINT
    }

    fun getRiEndpoint(): String? {
        return endpoints?.riEndpoint
    }

    fun getMraidEndpoint(): String? {
        return endpoints?.mraidEndpoint
    }

    fun getMraidJsVersion(): String {
        return getMraidEndpoint()?.let {
            "mraid_${Uri.parse(it).lastPathSegment}"
        } ?: "mraid_1"
    }

    fun getGDPRConsentMessage(): String? {
        return config?.userPrivacy?.gdpr?.consentMessage
    }

    fun getGDPRConsentTitle(): String? {
        return config?.userPrivacy?.gdpr?.consentTitle
    }

    fun getGDPRButtonAccept(): String? {
        return config?.userPrivacy?.gdpr?.buttonAccept
    }

    fun getGDPRButtonDeny(): String? {
        return config?.userPrivacy?.gdpr?.buttonDeny
    }

    fun getGDPRConsentMessageVersion(): String {
        return config?.userPrivacy?.gdpr?.consentMessageVersion ?: ""
    }

    fun getGDPRIsCountryDataProtected(): Boolean {
        return config?.userPrivacy?.gdpr?.isCountryDataProtected ?: false
    }

    fun getTcfStatus(): ConfigPayload.IABSettings.TcfStatus? {
        return ConfigPayload.IABSettings.TcfStatus.fromRawValue(config?.userPrivacy?.iab?.tcfStatus)
    }

    fun shouldDisableAdId(): Boolean {
        return config?.disableAdId ?: true
    }

    fun isReportIncentivizedEnabled(): Boolean {
        return config?.isReportIncentivizedEnabled ?: false
    }

    fun getConfigExtension(): String {
        return configExt ?: ""
    }

    fun configLastValidatedTimestamp(): Long {
        return config?.configLastValidatedTimestamp ?: CONFIG_LAST_VALIDATE_TS_DEFAULT
    }

    fun omEnabled() = config?.viewAbility?.om ?: false

    fun getMetricsEndpoint(): String {
        return endpoints?.metricsEndpoint.takeUnless { it.isNullOrEmpty() } ?: DEFAULT_METRICS_ENDPOINT
    }

    fun getErrorLoggingEndpoint(): String {
        return endpoints?.errorLogsEndpoint.takeUnless { it.isNullOrEmpty() } ?: DEFAULT_ERROR_LOGS_ENDPOINT
    }

    fun getMetricsEnabled(): Boolean {
        return config?.logMetricsSettings?.metricsEnabled ?: false
    }

    fun getLogLevel(): Int {
        return config?.logMetricsSettings?.errorLogLevel
            ?: AnalyticsClient.LogLevel.ERROR_LOG_LEVEL_ERROR.level
    }

    fun getSessionTimeout(): Long {
        return (config?.sessionTimeout ?: DEFAULT_SESSION_TIMEOUT_SECONDS) * 1000L
    }

    fun getSignalsSessionTimeout(): Long {
        return (config?.signalSessionTimeout ?: DEFAULT_SIGNALS_SESSION_TIMEOUT_SECONDS) * 1000L
    }

    fun rtaDebuggingEnabled(): Boolean {
        return config?.rtaDebugging ?: false
    }

    fun isCacheableAssetsRequired(): Boolean {
        return config?.isCacheableAssetsRequired ?: false
    }

    fun signalsDisabled(): Boolean {
        return config?.signalsDisabled ?: false
    }

    fun fpdEnabled(): Boolean {
        return config?.fpdEnabled ?: true
    }

    fun allowAutoRedirects(): Boolean = config?.autoRedirect?.allowAutoRedirect ?: false
    fun afterClickDuration(): Long = config?.autoRedirect?.afterClickDuration ?: Long.MAX_VALUE

    @VisibleForTesting
    internal fun checkConfigPayload(configPayload: ConfigPayload?): Int {
        return when {
            configPayload == null -> CONFIG_NOT_AVAILABLE
            configPayload.configLastValidatedTimestamp == null ||
                    configPayload.configLastValidatedTimestamp == CONFIG_LAST_VALIDATE_TS_DEFAULT -> CONFIG_NOT_AVAILABLE
            configPayload.endpoints == null -> CONFIG_LAST_VALIDATED_TIMESTAMP_ONLY
            else -> CONFIG_ALL_DATA
        }
    }

    @VisibleForTesting
    internal fun validateConfig(configPayload: ConfigPayload?): Boolean {
        return !(configPayload?.endpoints == null || !validateEndpoints(configPayload.endpoints)
                || configPayload.placements == null)
    }

    @VisibleForTesting
    internal fun validateEndpoints(endpoints: ConfigPayload.Endpoints? = this.endpoints): Boolean {
        var valid = true
        if (endpoints?.adsEndpoint.isNullOrEmpty()) {
            AnalyticsClient.logError(
                VungleError.INVALID_ADS_ENDPOINT,
                "The ads endpoint was not provided in the config."
            )
            valid = false
        }
        if (endpoints?.riEndpoint.isNullOrEmpty()) {
            AnalyticsClient.logError(
                VungleError.INVALID_RI_ENDPOINT,
                "The ri endpoint was not provided in the config."
            )
        }
        if (endpoints?.mraidEndpoint.isNullOrEmpty()) {
            AnalyticsClient.logError(
                VungleError.MRAID_DOWNLOAD_JS_ERROR,
                "The mraid endpoint was not provided in the config."
            )
            valid = false
        }
        if (endpoints?.metricsEndpoint.isNullOrEmpty()) {
            AnalyticsClient.logError(
                VungleError.INVALID_METRICS_ENDPOINT,
                "The metrics endpoint was not provided in the config."
            )
        }
        if (endpoints?.errorLogsEndpoint.isNullOrEmpty()) {
            /* uncomment this if there is a fallback errorLogsEndpoint or logged errors are stored
            to send when valid endpoint is provided
            AnalyticsClient.logError(
                VungleError.INVALID_LOG_ERROR_ENDPOINT,
                "The error logging endpoint was not provided in the config."
            )
            */
            Logger.e(TAG, "The error logging endpoint was not provided in the config.")
        }
        return valid
    }

    fun isCleverCacheEnabled(): Boolean {
        return config?.cleverCache?.enabled ?: false
    }

    fun getCleverCacheDiskSize(): Long {
        return config?.cleverCache?.diskSize?.also { return it * 1024 * 1024 }
            ?: (1000 * 1024 * 1024)
    }

    fun getCleverCacheDiskPercentage(): Int {
        return config?.cleverCache?.diskPercentage ?: 3
    }

    fun getCachedConfig(filePreferences: FilePreferences, appId: String): ConfigPayload? {
        try {
            val localAppId = filePreferences.getString(Cookie.CONFIG_APP_ID)
            if (localAppId.isNullOrEmpty() || !localAppId.equals(appId, ignoreCase = true)) {
                Logger.w(TAG, "app id mismatch, re-config")
                return null
            }

            filePreferences.getString(Cookie.CONFIG_RESPONSE)?.let {
                val configLastUpdateTime = filePreferences.getLong(Cookie.CONFIG_UPDATE_TIME, 0)
                val localConfig = json.decodeFromString<ConfigPayload>(it)
                if ((localConfig.configSettings?.refreshTime
                        ?: -1) + configLastUpdateTime < System.currentTimeMillis()
                ) {
                    Logger.w(TAG, "cache config expired. re-config")
                    return null
                }
                Logger.w(TAG, "use cache config.")

                return localConfig
            }
        } catch (e: Exception) {
            Logger.e(TAG, "Error while parsing cached config: ${e.message}")
            return null
        }
        return null
    }

    fun updateCachedConfig(
        config: ConfigPayload,
        filePreferences: FilePreferences,
    ) {
        try {
            filePreferences.put(Cookie.CONFIG_APP_ID, applicationId)
            filePreferences.put(Cookie.CONFIG_UPDATE_TIME, System.currentTimeMillis())
            filePreferences.put(Cookie.CONFIG_RESPONSE, json.encodeToString(config))
            filePreferences.apply()
        } catch (e: Exception) {
            Logger.e(TAG, "Exception: ${e.message} for updating cached config")
        }
    }

    @VisibleForTesting
    internal fun clearConfig() {
        this.endpoints = null
        this.placements = null
        this.config = null
    }
}
