package com.vungle.ads.internal.executor

import com.vungle.ads.OutOfMemory
import com.vungle.ads.internal.task.PriorityRunnable
import com.vungle.ads.internal.util.Logger
import java.util.concurrent.BlockingQueue
import java.util.concurrent.Callable
import java.util.concurrent.Future
import java.util.concurrent.ThreadFactory
import java.util.concurrent.ThreadPoolExecutor
import java.util.concurrent.TimeUnit

class VungleThreadPoolExecutor(
    corePoolSize: Int,
    maximumPoolSize: Int,
    keepAliveTime: Long,
    unit: TimeUnit?,
    workQueue: BlockingQueue<Runnable?>?,
    threadFactory: ThreadFactory?
) : ThreadPoolExecutor(
    corePoolSize,
    maximumPoolSize,
    keepAliveTime,
    unit,
    workQueue,
    threadFactory
) {

    interface ComparableRunnable : Comparable<Any>, Runnable

    companion object {
        private const val TAG = "VungleThreadPool"

        private fun wrapRunnableWithFail(command: Runnable, fail: Runnable) {
            try {
                command.run()
            } catch (ignored: OutOfMemoryError) {
                fail.run()
            }
        }

        private fun getWrappedRunnableWithFail(
            command: Runnable,
            fail: Runnable,
        ): ComparableRunnable {
            if (command is PriorityRunnable) {
                return object : PriorityRunnable(), ComparableRunnable {
                    override val priority: Int
                        get() = command.priority

                    override fun run() {
                        wrapRunnableWithFail(command, fail)
                    }

                    override fun compareTo(other: Any): Int {
                        if (other is PriorityRunnable) {
                            val current = priority
                            val incoming = other.priority
                            return incoming.compareTo(current)
                        }
                        return 0
                    }
                }
            }
            return object : ComparableRunnable {
                override fun run() {
                    wrapRunnableWithFail(command, fail)
                }

                override fun compareTo(other: Any): Int {
                    if (command is PriorityRunnable) {
                        return command.compareTo(other)
                    }
                    return 0
                }
            }
        }

        private fun <T> getWrappedCallableWithFallback(
            command: Callable<T>,
            failFallback: () -> Unit,
        ): Callable<T> {
            return Callable<T> {
                try {
                    command.call()
                } catch (oom: OutOfMemoryError) {
                    failFallback()
                    null
                }
            }
        }
    }

    override fun execute(command: Runnable) {
        try {
            super.execute(getWrappedRunnableWithFail(command) {
                OutOfMemory("execute error").logErrorNoReturnValue()
            })
        } catch (ex: Exception) {
            Logger.e(TAG, "execute error: $ex")
        }
    }

    fun execute(command: Runnable, fail: Runnable) {
        try {
            super.execute(getWrappedRunnableWithFail(command, fail))
        } catch (ex: Exception) {
            Logger.e(TAG, "execute error with fail: $ex")
            fail.run()
        }
    }

    override fun submit(task: Runnable): Future<*> {
        return try {
            super.submit(getWrappedRunnableWithFail(task) {
                OutOfMemory("submit error").logErrorNoReturnValue()
            })
        } catch (ex: Exception) {
            Logger.e(TAG, "submit error: $ex")
            FutureResult<Any>(null)
        }
    }

    override fun <T> submit(task: Runnable, result: T): Future<T> {
        return try {
            super.submit(getWrappedRunnableWithFail(task) {
                OutOfMemory("submit error with error").logErrorNoReturnValue()
            }, result)
        } catch (ex: Exception) {
            Logger.e(TAG, "submit error with result: $ex")
            return FutureResult<T>(null) as Future<T>
        }
    }

    fun submit(task: Runnable, fail: Runnable): Future<*> {
        return try {
            super.submit(getWrappedRunnableWithFail(task, fail))
        } catch (ex: Exception) {
            Logger.e(TAG, "submit error with fail: $ex")
            fail.run()
            FutureResult<Any>(null)
        }
    }

    override fun <T> submit(task: Callable<T>): Future<T> {
        return try {
            super.submit(getWrappedCallableWithFallback(task) {
                OutOfMemory("submit callable error").logErrorNoReturnValue()
            })
        } catch (ex: Exception) {
            Logger.e(TAG, "submit callable: $ex")
            return FutureResult<T>(null) as Future<T>
        }
    }

    init {
        allowCoreThreadTimeOut(true)
    }
}
