package com.vungle.ads

import android.content.Context
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.util.ViewUtility

/**
 * The size of a banner, MREC and inline ad.
 */
class VungleAdSize internal constructor(val width: Int, val height: Int) {

    internal var isAdaptiveHeight: Boolean = false

    internal var isAdaptiveWidth: Boolean = false

    companion object {

        @JvmField
        val BANNER = VungleAdSize(320, 50)

        @JvmField
        val BANNER_SHORT = VungleAdSize(300, 50)

        @JvmField
        val BANNER_LEADERBOARD = VungleAdSize(728, 90)

        @JvmField
        val MREC = VungleAdSize(300, 250)

        /**
         * Get the banner size for the given width. The max height is the device height according
         * to current device orientation. The final height is a fixed height from server side.
         * The width should be a fixed width. If the width is 0, the final width is a fixed width
         * from server side.
         */
        @JvmStatic
        fun getAdSizeWithWidth(context: Context, width: Int): VungleAdSize {
            val (_, deviceHeight) = ViewUtility.getDeviceWidthAndHeightWithOrientation(
                context,
                0
            )
            val w = if (width < 0) 0 else width
            return VungleAdSize(w, deviceHeight).also {
                if (it.width == 0) {
                    it.isAdaptiveWidth = true
                }
                it.isAdaptiveHeight = true
            }
        }

        /**
         * Get the banner size for the given width and max height. The final height is fixed height
         * from server side. The width should be a fixed width. If the width is 0, the final width
         * is a fixed width from server side.
         */
        @JvmStatic
        fun getAdSizeWithWidthAndMaxHeight(width: Int, maxHeight: Int): VungleAdSize {
            val w = if (width < 0) 0 else width
            val h = if (maxHeight < 0) 0 else maxHeight
            return VungleAdSize(w, h).also {
                if (it.width == 0) {
                    it.isAdaptiveWidth = true
                }
                it.isAdaptiveHeight = true
            }
        }

        /**
         * Get the banner size for the given width and height.
         * The height should be a fixed height. If it is 0, the final height is a fixed width from server side.
         * The width should be a fixed width. If it is 0, the final width is a fixed width from server side.
         */
        @JvmStatic
        fun getAdSizeWithWidthAndHeight(width: Int, height: Int): VungleAdSize {
            val w = if (width < 0) 0 else width
            val h = if (height < 0) 0 else height
            return VungleAdSize(w, h).also {
                if (it.width == 0) {
                    it.isAdaptiveWidth = true
                }
                if (it.height == 0) {
                    it.isAdaptiveHeight = true
                }
            }
        }

        /**
         * Get the banner size for the given width and height and placement reference id.
         */
        @JvmStatic
        fun getValidAdSizeFromSize(width: Int, height: Int, placementId: String): VungleAdSize {
            ConfigManager.getPlacement(placementId)?.takeIf { it.isInline() }?.let {
                return getAdSizeWithWidthAndHeight(width, height)
            }

            if (width >= MREC.width && height >= MREC.height) {
                return MREC
            } else if (width >= BANNER_LEADERBOARD.width && height >= BANNER_LEADERBOARD.height) {
                return BANNER_LEADERBOARD
            } else if (width >= BANNER.width && height >= BANNER.height) {
                return BANNER
            } else if (width >= BANNER_SHORT.width && height >= BANNER_SHORT.height) {
                return BANNER_SHORT
            }

            return getAdSizeWithWidthAndHeight(width, height)

        }

    }

    /**
     * Check if the ad size is valid.
     */
    internal fun isValidSize(): Boolean {
        return width >= 0 && height >= 0
    }

    override fun toString(): String {
        return "VungleAdSize(width=$width, height=$height)"
    }
}

@Deprecated("Use VungleAdSize instead")
enum class BannerAdSize(val sizeName: String, val width: Int, val height: Int) {
    VUNGLE_MREC("mrec", 300, 250),
    BANNER("banner", 320, 50),
    BANNER_SHORT("banner_short", 300, 50),
    BANNER_LEADERBOARD("banner_leaderboard", 728, 90);
}
