package com.vungle.ads

import android.content.Context
import android.content.IntentFilter
import android.media.AudioManager
import android.view.MotionEvent
import android.view.View
import android.view.ViewGroup
import android.widget.RelativeLayout
import com.vungle.ads.internal.AdInternal
import com.vungle.ads.internal.BannerAdImpl
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.ImpressionTracker
import com.vungle.ads.internal.executor.Executors
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.model.Placement
import com.vungle.ads.internal.omsdk.OMTracker
import com.vungle.ads.internal.platform.Platform
import com.vungle.ads.internal.presenter.AdEventListener
import com.vungle.ads.internal.presenter.MRAIDPresenter
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.ui.VungleWebClient
import com.vungle.ads.internal.ui.WatermarkView
import com.vungle.ads.internal.ui.view.MRAIDAdWidget
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.RingerModeReceiver
import com.vungle.ads.internal.util.ViewUtility
import java.util.concurrent.atomic.AtomicBoolean

class VungleBannerView(context: Context, val placementId: String, val adSize: VungleAdSize) :
    RelativeLayout(context) {

    private val ringerModeReceiver: RingerModeReceiver = RingerModeReceiver()

    var adListener: BannerAdListener? = null

    val adConfig
        get() = adViewImpl.adConfig

    val creativeId
        get() = adViewImpl.creativeId

    val eventId
        get() = adViewImpl.eventId

    private val adViewImpl = BannerAdImpl(context, placementId, adSize, AdConfig())

    private var calculatedPixelWidth = 0
    private var calculatedPixelHeight = 0

    private var adWidget: MRAIDAdWidget? = null
    private var presenter: MRAIDPresenter? = null
    private var imageView: WatermarkView? = null

    private var isOnImpressionCalled: Boolean = false
    private val destroyed = AtomicBoolean(false)
    private val presenterStarted = AtomicBoolean(false)

    private val isAdDownloaded = AtomicBoolean(false)
    private val isAdAttachedToWindow = AtomicBoolean(false)

    private val isInvisibleLogged = AtomicBoolean(false)

    private val impressionTracker by lazy { ImpressionTracker(context) }

    private var isReceiverRegistered = false

    fun getAdViewSize(): VungleAdSize {
        return adViewImpl.getAdViewSize()
    }

    fun load(adMarkup: String? = null) {
        adViewImpl.load(adMarkup)
    }

    fun finishAd() {
        finishAdInternal(true)
    }

    override fun onWindowVisibilityChanged(visibility: Int) {
        super.onWindowVisibilityChanged(visibility)

        setAdVisibility(visibility == VISIBLE)
    }

    override fun onAttachedToWindow() {
        super.onAttachedToWindow()
        Logger.d(TAG, "onAttachedToWindow(): ${hashCode()}")
        isAdAttachedToWindow.set(true)
        val placement = adViewImpl.adInternal.placement
        if (placement != null) {
            try {
                if (!isReceiverRegistered) {
                    val filter = IntentFilter(AudioManager.RINGER_MODE_CHANGED_ACTION)
                    context.registerReceiver(ringerModeReceiver, filter)
                    isReceiverRegistered = true
                    Logger.d(TAG, "registerReceiver(): ${ringerModeReceiver.hashCode()}")
                }
            } catch (e: Exception) {
                Logger.e(TAG, "registerReceiver error: ${e.localizedMessage}")
            }
        }

        renderAd()
    }

    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        Logger.d(TAG, "onDetachedFromWindow(): ${hashCode()}")
        isAdAttachedToWindow.set(false)
        val placement = adViewImpl.adInternal.placement
        if (placement != null) {
            try {
                if (isReceiverRegistered) {
                    context.unregisterReceiver(ringerModeReceiver)
                    isReceiverRegistered = false
                }
            } catch (e: Exception) {
                Logger.e(TAG, "unregisterReceiver error: ${e.localizedMessage}")
            }
        }
    }

    private fun setAdVisibility(isVisible: Boolean) {
        if (!isOnImpressionCalled) {
            return
        }

        // The visibility change must be triggered after impression fired.
        if (!destroyed.get()) {
            presenter?.setAdVisibility(isVisible)
        }
    }

    private fun checkHardwareAcceleration() {
        Logger.w(TAG, "hardwareAccelerated = $isHardwareAccelerated")
        if (!isHardwareAccelerated) {
            AnalyticsClient.logMetric(
                Sdk.SDKMetric.SDKMetricType.HARDWARE_ACCELERATE_DISABLED,
                logEntry = adViewImpl.logEntry
            )
        }
    }

    private fun logViewInvisibleOnPlay() {
        if (!isInvisibleLogged.getAndSet(true)) {
            Logger.d(
                TAG,
                "ImpressionTracker checked the banner view invisible on play. ${hashCode()}"
            )
            AnalyticsClient.logMetric(
                SingleValueMetric(Sdk.SDKMetric.SDKMetricType.VIEW_NOT_VISIBLE_ON_PLAY),
                adViewImpl.logEntry
            )
        }
    }

    private fun finishAdInternal(isFinishedByApi: Boolean) {
        if (destroyed.get()) {
            return
        }
        destroyed.set(true)
        val flag = (MRAIDAdWidget.AdStopReason.IS_AD_FINISHING
                or if (isFinishedByApi) MRAIDAdWidget.AdStopReason.IS_AD_FINISHED_BY_API else 0)
        presenter?.stop()
        presenter?.detach(flag)
        impressionTracker.destroy()
        try {
            parent?.let {
                if (it is ViewGroup) {
                    it.removeView(this)
                }
            }
            removeAllViews()
        } catch (e: Exception) {
            Logger.d(TAG, "Removing webView error: $e")
        }
    }

    private fun renderAd() {
        if (destroyed.get()) {
            Logger.w(TAG, "renderAd() - destroyed")
            return
        }

        if (!isAdDownloaded.get()) {
            Logger.d(TAG, "renderAd() - not ready: not downloaded.")
            return
        }

        if (!isAdAttachedToWindow.get()) {
            Logger.d(TAG, "renderAd() - not ready: not attached.")
            logViewInvisibleOnPlay()
            return
        }

        if (!presenterStarted.getAndSet(true)) {
            adViewImpl.adInternal.showToValidationMetric.markEnd()
            AnalyticsClient.logMetric(adViewImpl.adInternal.showToValidationMetric, logEntry = adViewImpl.logEntry)
            adViewImpl.adInternal.validationToPresentMetric.markStart()

            presenter?.prepare()

            impressionTracker.addView(this, object : ImpressionTracker.ImpressionListener {

                override fun onImpression(view: View?) {
                    Logger.d(TAG, "ImpressionTracker checked the banner view become visible.")
                    isOnImpressionCalled = true
                    checkHardwareAcceleration()
                    presenter?.start()
                }

                override fun onViewInvisible(view: View?) {
                    logViewInvisibleOnPlay()
                }
            })
        }

        if (adWidget != null && adWidget?.parent != this) {
            addView(adWidget, calculatedPixelWidth, calculatedPixelHeight)

            if (imageView != null) {
                addView(imageView, calculatedPixelWidth, calculatedPixelHeight)
                imageView?.bringToFront()
            }
        }

        val bannerLayoutParams = layoutParams
        if (bannerLayoutParams != null) {
            //Set Ad Size when Banner Ad is attached to Window
            bannerLayoutParams.height = calculatedPixelHeight
            bannerLayoutParams.width = calculatedPixelWidth
            requestLayout()
        }
    }

    private fun onBannerAdLoaded(baseAd: BaseAd) {
        AnalyticsClient.logMetric(
            SingleValueMetric(Sdk.SDKMetric.SDKMetricType.PLAY_AD_API), adViewImpl.logEntry
        )
        adViewImpl.adInternal.showToValidationMetric.markStart()
        val error = adViewImpl.adInternal.canPlayAd(true)
        if (error != null) {
            if (adViewImpl.adInternal.isErrorTerminal(error.code)) {
                adViewImpl.adInternal.adState = AdInternal.AdState.ERROR
            }
            adListener?.onAdFailedToPlay(baseAd, error)
            return
        }

        val advertisement = adViewImpl.adInternal.advertisement
        val placement = adViewImpl.adInternal.placement
        if (advertisement == null || placement == null) {
            adListener?.onAdFailedToPlay(
                baseAd,
                AdNotLoadedCantPlay("Ad or Placement is null")
                    .setLogEntry(adViewImpl.logEntry).logError()
            )
            return
        }

        adViewImpl.adInternal.cancelDownload()

        try {
            willPresentAdView(advertisement, placement, getAdViewSize())
        } catch (e: InstantiationException) {
            return
        }

        adViewImpl.responseToShowMetric.markEnd()
        AnalyticsClient.logMetric(adViewImpl.responseToShowMetric, adViewImpl.logEntry)

        adViewImpl.showToCloseMetric.markStart()
        adViewImpl.showToFailMetric.markStart()

        isAdDownloaded.set(true)

        adListener?.onAdLoaded(baseAd)

        renderAd()
    }

    @Throws(InstantiationException::class)
    private fun willPresentAdView(
        advertisement: AdPayload,
        placement: Placement,
        fixedAdSize: VungleAdSize
    ) {
        //set Ad Size
        calculatedPixelHeight = ViewUtility.dpToPixels(context, fixedAdSize.height)
        calculatedPixelWidth = ViewUtility.dpToPixels(context, fixedAdSize.width)
        val adEventListener = object : AdEventListener(
            adViewImpl.adPlayCallback,
            adViewImpl.adInternal.placement
        ) {}

        val adWidget: MRAIDAdWidget?
        try {
            adWidget = MRAIDAdWidget(context)
        } catch (ex: InstantiationException) {
            adEventListener.onError(
                AdCantPlayWithoutWebView().setLogEntry(adViewImpl.logEntry).logError(),
                placementId
            )
            throw ex
        }
        this.adWidget = adWidget

        adWidget.setCloseDelegate(object : MRAIDAdWidget.CloseDelegate {
            override fun close() {
                finishAdInternal(false)
            }
        })

        adWidget.setOnViewTouchListener(object : MRAIDAdWidget.OnViewTouchListener {
            override fun onTouch(event: MotionEvent?): Boolean {
                presenter?.onViewTouched(event)
                return false
            }
        })


        val executors: Executors by ServiceLocator.inject(context)
        val omTrackerFactory: OMTracker.Factory by ServiceLocator.inject(context)
        val omTracker =
            omTrackerFactory.make(ConfigManager.omEnabled() && advertisement.omEnabled())
        val platform: Platform by ServiceLocator.inject(context)
        val webClient = VungleWebClient(
            advertisement,
            placement,
            executors.offloadExecutor,
            platform = platform
        )

        ringerModeReceiver.webClient = webClient
        webClient.setWebViewObserver(omTracker)

        presenter =
            MRAIDPresenter(
                adWidget,
                advertisement,
                placement,
                webClient,
                executors.jobExecutor,
                omTracker,
                adViewImpl.adInternal.bidPayload,
                platform,
            ).apply {
                setEventListener(adEventListener)
            }

        adConfig.getWatermark()?.let {
            imageView = WatermarkView(context, it)
        }
    }

    companion object {
        private const val TAG = "VungleBannerView"
    }

    init {
        adViewImpl.adListener = object : BannerAdListener {
            override fun onAdLoaded(baseAd: BaseAd) {
                onBannerAdLoaded(baseAd)
            }

            override fun onAdStart(baseAd: BaseAd) {
                adListener?.onAdStart(baseAd)
            }

            override fun onAdImpression(baseAd: BaseAd) {
                adListener?.onAdImpression(baseAd)
            }

            override fun onAdEnd(baseAd: BaseAd) {
                adListener?.onAdEnd(baseAd)
            }

            override fun onAdClicked(baseAd: BaseAd) {
                adListener?.onAdClicked(baseAd)
            }

            override fun onAdLeftApplication(baseAd: BaseAd) {
                adListener?.onAdLeftApplication(baseAd)
            }

            override fun onAdFailedToLoad(baseAd: BaseAd, adError: VungleError) {
                adListener?.onAdFailedToLoad(baseAd, adError)
            }

            override fun onAdFailedToPlay(baseAd: BaseAd, adError: VungleError) {
                adListener?.onAdFailedToPlay(baseAd, adError)
            }

        }
    }

}
