package com.vungle.ads.internal.bidding

import android.content.Context
import androidx.annotation.VisibleForTesting
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.GzipEncodeError
import com.vungle.ads.JsonEncodeError
import com.vungle.ads.ServiceLocator
import com.vungle.ads.SingleValueMetric
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.model.RtbRequest
import com.vungle.ads.internal.model.RtbToken
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.network.VungleHeader
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.util.ActivityManager
import com.vungle.ads.internal.util.InputOutputUtils
import com.vungle.ads.internal.util.Logger
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json

internal class BidTokenEncoder(
    private val context: Context
) {

    private var bidTokenRequestedMetric =
        SingleValueMetric(Sdk.SDKMetric.SDKMetricType.BID_TOKEN_REQUESTED)

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal var ordinalView = 0

    @OptIn(ExperimentalSerializationApi::class)
    private val json = Json {
        ignoreUnknownKeys = true
        encodeDefaults = true
        explicitNulls = false
    }

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal var enterBackgroundTime = 0L

    companion object {
        private const val TAG = "BidTokenEncoder"

        internal const val TOKEN_VERSION = 6
    }

    init {
        ActivityManager.addLifecycleListener(object : ActivityManager.LifeCycleCallback() {
            override fun onForeground() {
                onResume()
            }

            override fun onBackground() {
                onPause()
            }
        })
    }

    data class BiddingTokenInfo(val bidToken: String, val errorMessage: String)

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal fun onResume() {
        Logger.d(TAG, "BidTokenEncoder#onForeground()")
        val currentTimeMillis = System.currentTimeMillis()
        val sessionTimeout = ConfigManager.getSessionTimeout()
        if (currentTimeMillis > enterBackgroundTime + sessionTimeout) {
            // Reset ordinal view count
            ordinalView = 0
            enterBackgroundTime = 0L
        }
    }

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal fun onPause() {
        Logger.d(TAG, "BidTokenEncoder#onBackground()")
        enterBackgroundTime = System.currentTimeMillis()
    }

    fun encode(): BiddingTokenInfo {
        ordinalView++
        return generateBidToken()
    }

    private fun generateBidToken(): BiddingTokenInfo {
        val tokenData: String?
        AnalyticsClient.logMetric(bidTokenRequestedMetric)
        try {
            tokenData = constructV6Token()
        } catch (e: Exception) {
            val error = "Failed to encode TokenParameters. ${e.localizedMessage}"
            JsonEncodeError(error).logErrorNoReturnValue()
            return BiddingTokenInfo("", error)
        }

        Logger.d(TAG, "BidToken: $tokenData")

        try {
            val converted = InputOutputUtils.convertForSending(tokenData)
            val bidToken = "$TOKEN_VERSION:$converted"
            Logger.d(TAG, "After conversion: $bidToken")
            return BiddingTokenInfo(bidToken, "")
        } catch (e: Exception) {
            val error = "Fail to gzip token data. ${e.localizedMessage}"
            GzipEncodeError(error).logErrorNoReturnValue()
            return BiddingTokenInfo("", error)
        }
    }

    @VisibleForTesting
    internal fun constructV6Token(): String {
        val vungleApiClient: VungleApiClient by ServiceLocator.inject(context)
        val commonRequestBody = vungleApiClient.requestBody(
            !ConfigManager.signalsDisabled(),
            ConfigManager.fpdEnabled()
        )
        val body = RtbToken(
            commonRequestBody.device,
            commonRequestBody.user,
            commonRequestBody.ext,
            RtbRequest(VungleHeader.headerUa),
            ordinalView
        )
        return json.encodeToString(body)
    }

}
