package com.vungle.ads.internal.executor

import androidx.annotation.VisibleForTesting
import com.vungle.ads.OutOfMemory
import com.vungle.ads.internal.task.PriorityRunnable
import com.vungle.ads.internal.util.Logger
import java.util.concurrent.BlockingQueue
import java.util.concurrent.Callable
import java.util.concurrent.Future
import java.util.concurrent.ThreadPoolExecutor
import java.util.concurrent.TimeUnit

class VungleThreadPoolExecutor(
    corePoolSize: Int,
    maximumPoolSize: Int,
    keepAliveTime: Long,
    unit: TimeUnit?,
    workQueue: BlockingQueue<Runnable?>?,
    val threadFactory: NamedThreadFactory?
) : ThreadPoolExecutor(
    corePoolSize,
    maximumPoolSize,
    keepAliveTime,
    unit,
    workQueue,
    threadFactory
) {

    interface ComparableRunnable : Comparable<Any>, Runnable

    companion object {
        private const val TAG = "VungleThreadPool"

        private fun wrapRunnableWithFail(command: Runnable, fail: Runnable) {
            try {
                command.run()
            } catch (ignored: OutOfMemoryError) {
                fail.run()
            }
        }

        private fun getWrappedRunnableWithFail(
            command: Runnable,
            fail: Runnable,
        ): ComparableRunnable {
            if (command is PriorityRunnable) {
                return object : PriorityRunnable(), ComparableRunnable {
                    override val priority: Int
                        get() = command.priority

                    override fun run() {
                        wrapRunnableWithFail(command, fail)
                    }

                    override fun compareTo(other: Any): Int {
                        if (other is PriorityRunnable) {
                            val current = priority
                            val incoming = other.priority
                            return incoming.compareTo(current)
                        }
                        return 0
                    }
                }
            }
            return object : ComparableRunnable {
                override fun run() {
                    wrapRunnableWithFail(command, fail)
                }

                override fun compareTo(other: Any): Int {
                    if (command is PriorityRunnable) {
                        return command.compareTo(other)
                    }
                    return 0
                }
            }
        }

        private fun <T> getWrappedCallableWithFallback(
            command: Callable<T>,
            failFallback: () -> Unit,
        ): Callable<T> {
            return Callable<T> {
                try {
                    command.call()
                } catch (oom: OutOfMemoryError) {
                    failFallback()
                    null
                }
            }
        }
    }

    private fun executorName() = threadFactory?.name ?: "VungleThreadPoolExecutor"

    override fun execute(command: Runnable) {
        try {
            super.execute(getWrappedRunnableWithFail(command) {
                OutOfMemory("execute error in ${executorName()}").logErrorNoReturnValue()
            })
        } catch (ex: Exception) {
            Logger.e(TAG, "execute exception", ex)
        } catch (err: OutOfMemoryError) {
            val msg = "execute error in ${executorName()}: ${err.localizedMessage}"
            Logger.e(TAG, msg, err)
            OutOfMemory(msg).logErrorNoReturnValue()
        }
    }

    fun execute(command: Runnable, fail: Runnable) {
        try {
            super.execute(getWrappedRunnableWithFail(command, fail))
        } catch (ex: Exception) {
            Logger.e(TAG, "execute exception with fail", ex)
            fail.run()
        } catch (err: OutOfMemoryError) {
            val msg = "execute error with fail in ${executorName()}: ${err.localizedMessage}"
            Logger.e(TAG, msg, err)
            OutOfMemory(msg).logErrorNoReturnValue()
            fail.run()
        }
    }

    override fun submit(task: Runnable): Future<*> {
        return try {
            super.submit(getWrappedRunnableWithFail(task) {
                OutOfMemory("submit error in ${executorName()}").logErrorNoReturnValue()
            })
        } catch (ex: Exception) {
            Logger.e(TAG, "submit exception", ex)
            FutureResult<Any>(null)
        } catch (err: OutOfMemoryError) {
            val msg = "submit error in ${executorName()}: ${err.localizedMessage}"
            Logger.e(TAG, msg, err)
            OutOfMemory(msg).logErrorNoReturnValue()
            FutureResult<Any>(null)
        }
    }

    override fun <T> submit(task: Runnable, result: T): Future<T> {
        return try {
            super.submit(getWrappedRunnableWithFail(task) {
                OutOfMemory("submit error with result in ${executorName()}").logErrorNoReturnValue()
            }, result)
        } catch (ex: Exception) {
            Logger.e(TAG, "submit exception with result", ex)
            FutureResult<T>(null) as Future<T>
        } catch (err: OutOfMemoryError) {
            val msg = "submit error with result in ${executorName()}: ${err.localizedMessage}"
            Logger.e(TAG, msg, err)
            OutOfMemory(msg).logErrorNoReturnValue()
            FutureResult<T>(null) as Future<T>
        }
    }

    internal fun submit(task: Runnable, fail: Runnable): Future<*> {
        return try {
            super.submit(getWrappedRunnableWithFail(task, fail))
        } catch (ex: Exception) {
            Logger.e(TAG, "submit exception with fail", ex)
            fail.run()
            FutureResult<Any>(null)
        } catch (err: OutOfMemoryError) {
            val msg = "submit error with fail in ${executorName()}: ${err.localizedMessage}"
            Logger.e(TAG, msg, err)
            OutOfMemory(msg).logErrorNoReturnValue()
            fail.run()
            FutureResult<Any>(null)
        }
    }

    override fun <T> submit(task: Callable<T>): Future<T> {
        return try {
            super.submit(getWrappedCallableWithFallback(task) {
                OutOfMemory("submit callable error in ${executorName()}").logErrorNoReturnValue()
            })
        } catch (ex: Exception) {
            Logger.e(TAG, "submit exception callable: $ex")
            FutureResult<T>(null) as Future<T>
        } catch (err: OutOfMemoryError) {
            val msg = "submit error callable in ${executorName()}: ${err.localizedMessage}"
            Logger.e(TAG, msg, err)
            OutOfMemory(msg).logErrorNoReturnValue()
            FutureResult<T>(null) as Future<T>
        }
    }

    init {
        allowCoreThreadTimeOut(true)
    }
}
