@file:JvmName("BaseAdLoader")

package com.vungle.ads.internal.load

import android.content.Context
import androidx.annotation.WorkerThread
import com.vungle.ads.AdExpiredError
import com.vungle.ads.AdPayloadError
import com.vungle.ads.AdResponseEmptyError
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.AssetDownloadError
import com.vungle.ads.AssetRequestError
import com.vungle.ads.AssetResponseDataError
import com.vungle.ads.IndexHtmlError
import com.vungle.ads.InvalidAssetUrlError
import com.vungle.ads.InvalidEventIdError
import com.vungle.ads.InvalidTemplateURLError
import com.vungle.ads.MraidJsError
import com.vungle.ads.NativeAssetError
import com.vungle.ads.OmSdkJsError
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.SingleValueMetric
import com.vungle.ads.TemplateUnzipError
import com.vungle.ads.TimeIntervalMetric
import com.vungle.ads.VungleError
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.Constants
import com.vungle.ads.internal.NativeAdInternal
import com.vungle.ads.internal.downloader.AssetDownloadListener
import com.vungle.ads.internal.downloader.AssetDownloadListener.DownloadError.Companion.DEFAULT_SERVER_CODE
import com.vungle.ads.internal.downloader.DownloadRequest
import com.vungle.ads.internal.downloader.Downloader
import com.vungle.ads.internal.executor.Executors
import com.vungle.ads.internal.model.AdAsset
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.network.TpatSender
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.omsdk.OMInjector
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.protos.Sdk.SDKError
import com.vungle.ads.internal.signals.SignalManager
import com.vungle.ads.internal.util.FileUtility
import com.vungle.ads.internal.util.LogEntry
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.PathProvider
import com.vungle.ads.internal.util.UnzipUtility
import com.vungle.ads.internal.util.Utils
import java.io.File
import java.io.IOException
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicLong


abstract class BaseAdLoader(
    val context: Context,
    val vungleApiClient: VungleApiClient,
    val sdkExecutors: Executors,
    private val omInjector: OMInjector,
    private val downloader: Downloader,
    val pathProvider: PathProvider,
    val adRequest: AdRequest
) {

    companion object {
        private const val TAG = "BaseAdLoader"
        private const val DOWNLOADED_FILE_NOT_FOUND = "Downloaded file not found!"
    }

    private val downloadCount: AtomicLong = AtomicLong(0)

    private val downloadRequiredCount: AtomicLong = AtomicLong(0)

    private var adLoaderCallback: AdLoaderCallback? = null

    private var notifySuccess = AtomicBoolean(false)
    private var notifyFailed = AtomicBoolean(false)

    private val adAssets: MutableList<AdAsset> = mutableListOf()

    internal var advertisement: AdPayload? = null
    private var fullyDownloaded = AtomicBoolean(true)
    private var requiredAssetDownloaded = AtomicBoolean(true)

    private var mainVideoSizeMetric =
        SingleValueMetric(Sdk.SDKMetric.SDKMetricType.ASSET_FILE_SIZE)

    private var templateSizeMetric =
        SingleValueMetric(Sdk.SDKMetric.SDKMetricType.TEMPLATE_ZIP_SIZE)

    private var templateHtmlSizeMetric =
        SingleValueMetric(Sdk.SDKMetric.SDKMetricType.TEMPLATE_HTML_SIZE)

    private var assetDownloadDurationMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.ASSET_DOWNLOAD_DURATION_MS)

    private var adRequiredDownloadDurationMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.AD_REQUIRED_DOWNLOAD_DURATION_MS)

    private var adOptionalDownloadDurationMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.AD_OPTIONAL_DOWNLOAD_DURATION_MS)

    internal var logEntry: LogEntry? = null

    // check if all the assets downloaded successfully.
    private val assetDownloadListener: AssetDownloadListener
        get() {
            return object : AssetDownloadListener {
                override fun onError(
                    error: AssetDownloadListener.DownloadError?,
                    downloadRequest: DownloadRequest
                ) {
                    Logger.e(TAG, "onError called: reason ${error?.reason}; cause ${error?.cause}")
                    sdkExecutors.backgroundExecutor.execute {

                        fullyDownloaded.set(false)
                        if (downloadRequest.asset.isRequired) {
                            requiredAssetDownloaded.set(false)
                        }

                        val errorMsg =
                            "Failed to download assets. required=${downloadRequest.asset.isRequired} " +
                                    "reason=${error?.reason} cause=${error?.cause}"
                        if (downloadRequest.asset.isRequired && downloadRequiredCount.decrementAndGet() <= 0) {
                            // call failure callback if all required assets are not downloaded.
                            onAdLoadFailed(
                                AssetDownloadError(
                                    SDKError.Reason.ASSET_RESPONSE_DATA_ERROR,
                                    errorMsg
                                ).setLogEntry(logEntry).logError()
                            )
                            // cancel the rest of optional download requests
                            cancel()
                            return@execute
                        }

                        if (downloadCount.decrementAndGet() <= 0) {
                            onAdLoadFailed(
                                AssetDownloadError(
                                    SDKError.Reason.ASSET_RESPONSE_DATA_ERROR,
                                    errorMsg
                                ).setLogEntry(logEntry).logError()
                            )
                        }
                    }
                }

                override fun onSuccess(file: File, downloadRequest: DownloadRequest) {
                    sdkExecutors.backgroundExecutor.execute {
                        if (!file.exists()) {
                            onError(
                                AssetDownloadListener.DownloadError(
                                    DEFAULT_SERVER_CODE,
                                    IOException(DOWNLOADED_FILE_NOT_FOUND),
                                    AssetDownloadListener.DownloadError.ErrorReason.FILE_NOT_FOUND_ERROR
                                ),
                                downloadRequest
                            ) //AdAsset table will be updated in onError callback
                            return@execute
                        }

                        val adAsset = downloadRequest.asset
                        adAsset.fileSize = file.length()
                        adAsset.status = AdAsset.Status.DOWNLOAD_SUCCESS

                        if (downloadRequest.isTemplate) {
                            downloadRequest.stopRecord()
                            val templateFileSizeMetric = if (downloadRequest.isHtmlTemplate) {
                                templateHtmlSizeMetric
                            } else {
                                templateSizeMetric
                            }
                            templateFileSizeMetric.value = file.length()
                            AnalyticsClient.logMetric(
                                templateFileSizeMetric,
                                logEntry,
                                metaData = adAsset.serverPath
                            )
                        } else if (downloadRequest.isMainVideo) {
                            mainVideoSizeMetric.value = file.length()
                            AnalyticsClient.logMetric(
                                mainVideoSizeMetric,
                                logEntry,
                                metaData = adAsset.serverPath
                            )
                        }

                        // Update the asset path in the mraid file map
                        advertisement?.updateAdAssetPath(adAsset)

                        if (downloadRequest.isTemplate) {
                            if (!processVmTemplate(adAsset, advertisement)) {
                                fullyDownloaded.set(false)
                                if (adAsset.isRequired) {
                                    requiredAssetDownloaded.set(false)
                                }
                            }
                        }

                        if (adAsset.isRequired && downloadRequiredCount.decrementAndGet() <= 0) {
                            // onAdLoaded callback will be triggered when isRequired assets are downloaded.
                            if (requiredAssetDownloaded.get()) {
                                onRequiredDownloadCompleted()
                            } else {
                                onAdLoadFailed(
                                    AssetDownloadError(
                                        SDKError.Reason.ASSET_RESPONSE_DATA_ERROR,
                                        "Failed to download required assets."
                                    ).setLogEntry(logEntry).logError()
                                )
                                cancel()
                                return@execute
                            }
                        }

                        // check if all the assets downloaded successfully.
                        if (downloadCount.decrementAndGet() <= 0) {
                            // set advertisement state to READY
                            if (fullyDownloaded.get()) {
                                onDownloadCompleted(adRequest)
                            } else {
                                onAdLoadFailed(
                                    AssetDownloadError(
                                        SDKError.Reason.ASSET_RESPONSE_DATA_ERROR,
                                        "Failed to download assets."
                                    ).setLogEntry(logEntry).logError()
                                )
                            }
                        }
                    }
                }
            }
        }

    fun loadAd(adLoaderCallback: AdLoaderCallback) {
        this.adLoaderCallback = adLoaderCallback

        sdkExecutors.backgroundExecutor.execute {
            requestAd()
        }
    }

    protected abstract fun requestAd()

    abstract fun onAdLoadReady()

    fun cancel() {
        downloader.cancelAll()
    }

    private fun downloadAssets() {
        assetDownloadDurationMetric.markStart()
        adRequiredDownloadDurationMetric.markStart()
        adOptionalDownloadDurationMetric.markStart()
        downloadCount.set(adAssets.size.toLong())
        downloadRequiredCount.set(adAssets.filter { it.isRequired }.size.toLong())
        for (asset in adAssets) {
            val downloadRequest = DownloadRequest(getAssetPriority(asset), asset, logEntry)
            if (downloadRequest.isTemplate) {
                downloadRequest.startRecord()
            }
            downloader.download(downloadRequest, assetDownloadListener)
        }
    }

    fun onAdLoadFailed(error: VungleError) {
        if (!notifySuccess.get() && notifyFailed.compareAndSet(false, true)) {
            adLoaderCallback?.onFailure(error)
        }
    }

    private fun onAdReady() {
        advertisement?.let {

            // onSuccess can only be called once. For ADO case, it will be called right after
            // template is downloaded. After all assets are downloaded, onSuccess will not be called.
            if (!notifyFailed.get() && notifySuccess.compareAndSet(false, true)) {
                // After real time ad loaded, will send win notifications.
                // Use this abstract method to notify sub ad loader doing some clean up job.
                onAdLoadReady()
                adLoaderCallback?.onSuccess(it)
            }
        }
    }

    private fun fileIsValid(file: File, adAsset: AdAsset): Boolean {
        return file.exists() && file.length() == adAsset.fileSize
    }

    private fun unzipFile(
        downloadedFile: File,
        destinationDir: File
    ): Boolean {
        val existingPaths: MutableList<String> = ArrayList()
        for (asset in adAssets) {
            if (asset.fileType == AdAsset.FileType.ASSET) {
                existingPaths.add(asset.localPath)
            }
        }
        try {
            UnzipUtility.unzip(downloadedFile.path, destinationDir.path,
                object : UnzipUtility.Filter {
                    override fun matches(extractPath: String?): Boolean {
                        if (extractPath.isNullOrEmpty()) {
                            return true
                        }
                        val toExtract = File(extractPath)
                        for (existing in existingPaths) {
                            val existingFile = File(existing)
                            if (existingFile == toExtract)
                                return false
                            if (toExtract.path.startsWith(existingFile.path + File.separator))
                                return false
                        }
                        return true
                    }
                })

            val file = File(destinationDir.path, Constants.AD_INDEX_FILE_NAME)
            if (!file.exists()) {
                IndexHtmlError(
                    SDKError.Reason.INVALID_INDEX_URL,
                    "Failed to retrieve indexFileUrl from the Ad"
                ).setLogEntry(logEntry).logErrorNoReturnValue()
                return false
            }

        } catch (ex: Exception) {
            TemplateUnzipError("Unzip failed: ${ex.message}")
                .setLogEntry(logEntry).logErrorNoReturnValue()
            return false
        }

        FileUtility.delete(downloadedFile)
        return true
    }

    private fun getDestinationDir(advertisement: AdPayload): File? {
        return pathProvider.getDownloadsDirForAd(advertisement.eventId())
    }

    private fun injectMraidJS(destinationDir: File): Boolean {
        try {
            val adMraidJS = File(destinationDir.path, Constants.AD_MRAID_JS_FILE_NAME)
            val mraidJsPath = pathProvider.getJsAssetDir(ConfigManager.getMraidJsVersion())
            val mraidJsFile = File(mraidJsPath, Constants.MRAID_JS_FILE_NAME)
            if (mraidJsFile.exists()) {
                mraidJsFile.copyTo(adMraidJS, true)
            } else {
                MraidJsError(
                    SDKError.Reason.MRAID_JS_DOES_NOT_EXIST,
                    "mraid js source file not exist."
                )
                    .setLogEntry(logEntry).logErrorNoReturnValue()
                return false
            }
        } catch (e: Exception) {
            Logger.e(TAG, "Failed to inject mraid.js: ${e.message}")
            MraidJsError(
                SDKError.Reason.MRAID_JS_COPY_FAILED,
                "Failed to copy mraid js to ad folder: ${e.message}"
            )
                .setLogEntry(logEntry).logErrorNoReturnValue()
            return false
        }
        return true
    }

    private fun processVmTemplate(
        asset: AdAsset,
        advertisement: AdPayload?
    ): Boolean {
        if (advertisement == null) {
            return false
        }
        if (asset.status != AdAsset.Status.DOWNLOAD_SUCCESS) {
            return false
        }
        if (asset.localPath.isEmpty()) {
            return false
        }
        val vmTemplate = File(asset.localPath)
        if (!fileIsValid(vmTemplate, asset)) {
            return false
        }

        val destinationDir = getDestinationDir(advertisement)
        if (destinationDir == null || !destinationDir.isDirectory) {
            Logger.e(TAG, "Unable to access Destination Directory")
            return false
        }

        if (asset.fileType == AdAsset.FileType.ZIP && !unzipFile(vmTemplate, destinationDir)) {
            return false
        }

        // Inject OMSDK
        if (advertisement.omEnabled()) {
            try {
                omInjector.injectJsFiles(destinationDir)
            } catch (e: Exception) {
                Logger.e(TAG, "Failed to inject OMSDK: ${e.message}")
                OmSdkJsError(
                    SDKError.Reason.OMSDK_JS_WRITE_FAILED,
                    "Failed to inject OMSDK: ${e.message}"
                ).setLogEntry(logEntry).logErrorNoReturnValue()
            }
        }

        // Inject MRAID JS
        return injectMraidJS(destinationDir).also {
            FileUtility.printDirectoryTree(destinationDir)
        }

    }

    private fun onRequiredDownloadCompleted() {
        adRequiredDownloadDurationMetric.markEnd()
        AnalyticsClient.logMetric(adRequiredDownloadDurationMetric, logEntry)

        onAdReady()
    }

    @WorkerThread
    private fun onDownloadCompleted(request: AdRequest) {
        Logger.d(TAG, "All download completed $request")
        advertisement?.setAssetFullyDownloaded()
        onAdReady()

        assetDownloadDurationMetric.markEnd()
        AnalyticsClient.logMetric(assetDownloadDurationMetric, logEntry)

        adOptionalDownloadDurationMetric.markEnd()
        AnalyticsClient.logMetric(adOptionalDownloadDurationMetric, logEntry)
    }

    internal fun handleAdMetaData(advertisement: AdPayload, metric: SingleValueMetric? = null) {
        this.advertisement = advertisement
        // remember the log entry instance for the advertisement
        advertisement.logEntry = logEntry
        logEntry?.eventId = advertisement.eventId()
        logEntry?.creativeId = advertisement.getCreativeId()
        logEntry?.adSource = advertisement.getAdSource()

        // Update config
        advertisement.config()?.let { config ->
            ConfigManager.initWithConfig(context, config, false, metric)
        }

        val error = validateAdMetadata(advertisement)
        if (error != null) {
            onAdLoadFailed(error.setLogEntry(logEntry).logError())
            return
        }

        val destinationDir: File? = getDestinationDir(advertisement)
        if (destinationDir == null || !destinationDir.isDirectory || !destinationDir.exists()) {
            onAdLoadFailed(
                AssetDownloadError(
                    SDKError.Reason.ASSET_WRITE_ERROR,
                    "Invalid directory. $destinationDir"
                ).setLogEntry(logEntry).logError()
            )
            return
        }
        val signalManager: SignalManager by inject(context)

        // URLs for start to load ad notification when load ad after get the adm and
        // before assets start to download.
        advertisement.adUnit()?.loadAdUrls?.also { loadAdUrls ->
            val tpatSender = TpatSender(
                vungleApiClient,
                logEntry,
                sdkExecutors.ioExecutor,
                pathProvider,
                signalManager = signalManager
            )
            loadAdUrls.forEach {
                tpatSender.sendTpat(it, sdkExecutors.jobExecutor)
            }
        }

        if (adAssets.isNotEmpty()) {
            adAssets.clear()
        }
        adAssets.addAll(advertisement.getDownloadableAssets(destinationDir))

        if (adAssets.isEmpty()) {
            onAdLoadFailed(
                AssetDownloadError(
                    SDKError.Reason.INVALID_ASSET_URL,
                    "No assets to download."
                ).setLogEntry(logEntry).logError()
            )
            return
        }

        // Move mraid.js download here, before start downloading other assets.
        MraidJsLoader.downloadJs(
            pathProvider, downloader, sdkExecutors.backgroundExecutor,
            object : MraidJsLoader.DownloadResultListener {
                override fun onDownloadResult(downloadResult: Int) {
                    if (downloadResult == MraidJsLoader.MRAID_AVAILABLE
                        || downloadResult == MraidJsLoader.MRAID_DOWNLOADED
                    ) {
                        if (downloadResult == MraidJsLoader.MRAID_DOWNLOADED) {
                            AnalyticsClient.logMetric(
                                Sdk.SDKMetric.SDKMetricType.MRAID_DOWNLOAD_JS_RETRY_SUCCESS,
                                logEntry = logEntry
                            )
                        }
                        downloadAssets()
                    } else {
                        adLoaderCallback?.onFailure(
                            MraidJsError(
                                SDKError.Reason.MRAID_DOWNLOAD_JS_ERROR,
                                "Failed to download mraid.js."
                            )
                        )
                    }
                }
            }, advertisement
        )

    }

    private fun getAssetPriority(adAsset: AdAsset): DownloadRequest.Priority {
        return if (adAsset.isRequired) {
            DownloadRequest.Priority.CRITICAL
        } else {
            DownloadRequest.Priority.HIGHEST
        }
    }

    private fun validateAdMetadata(adPayload: AdPayload): VungleError? {
        adPayload.adUnit()?.sleep?.let {
            return getErrorInfo(adPayload)
        }

        if (adRequest.placement.referenceId != advertisement?.placementId()) {
            val description = "Requests and responses don't match ${advertisement?.placementId()}."
            return AdResponseEmptyError(description)
        }

        val templateSettingsError = getTemplateError(adPayload)
        if (templateSettingsError != null) {
            return templateSettingsError
        }

        if (adPayload.hasExpired()) {
            return AdExpiredError("The ad markup has expired for playback.")
        }

        if (adPayload.eventId().isNullOrEmpty()) {
            return InvalidEventIdError("Event id is invalid.")
        }

        return null
    }

    private fun getTemplateError(adPayload: AdPayload): VungleError? {
        val templateSettings = adPayload.adUnit()?.templateSettings
        if (templateSettings == null) {
            val description = "Missing template settings"
            return AssetResponseDataError(description)
        }

        val cacheableReplacements = templateSettings.cacheableReplacements
        if (adPayload.isNativeTemplateType()) {
            cacheableReplacements?.let {
                if (it[NativeAdInternal.TOKEN_MAIN_IMAGE]?.url == null) {
                    return NativeAssetError("Unable to load null main image.")
                }
                if (it[NativeAdInternal.TOKEN_VUNGLE_PRIVACY_ICON_URL]?.url == null) {
                    return NativeAssetError("Unable to load null privacy image.")
                }
            }
        } else {
            val templateUrl = adPayload.adUnit()?.templateURL
            val vmUrl = adPayload.adUnit()?.vmURL
            if (templateUrl.isNullOrEmpty() && vmUrl.isNullOrEmpty()) {
                val description = "Failed to prepare null vmURL or templateURL for downloading."
                return InvalidTemplateURLError(description)
            }

            if (!templateUrl.isNullOrEmpty() && !Utils.isUrlValid(templateUrl)) {
                val description = "Failed to load template: $templateUrl"
                return AssetRequestError(description)
            }

            if (!vmUrl.isNullOrEmpty() && !Utils.isUrlValid(vmUrl)) {
                val description = "Failed to load vm url: $vmUrl"
                return AssetRequestError(description)
            }
        }

        cacheableReplacements?.forEach {
            val httpUrl = it.value.url
            if (httpUrl.isNullOrEmpty()) {
                return InvalidAssetUrlError("Invalid asset URL $httpUrl")
            }
            if (!Utils.isUrlValid(httpUrl)) {
                return AssetRequestError("Invalid asset URL $httpUrl")
            }
        }

        return null
    }

    private fun getErrorInfo(adPayload: AdPayload): VungleError {
        val errorCode: Int? = adPayload.adUnit()?.errorCode
        val sleep = adPayload.adUnit()?.sleep
        val info = adPayload.adUnit()?.info
        val errorMsg = "Response error: $sleep, Request failed with error: $errorCode, $info"
        return when (errorCode) {
            Sdk.SDKError.Reason.AD_NO_FILL_VALUE,
            Sdk.SDKError.Reason.AD_LOAD_TOO_FREQUENTLY_VALUE,
            Sdk.SDKError.Reason.AD_SERVER_ERROR_VALUE,
            Sdk.SDKError.Reason.AD_PUBLISHER_MISMATCH_VALUE,
            Sdk.SDKError.Reason.AD_INTERNAL_INTEGRATION_ERROR_VALUE -> AdPayloadError(
                SDKError.Reason.forNumber(errorCode),
                errorMsg
            )

            else -> AdPayloadError(SDKError.Reason.PLACEMENT_SLEEP, errorMsg)
        }

    }
}
