package com.vungle.ads.internal.network

import com.vungle.ads.internal.util.Logger
import androidx.annotation.VisibleForTesting
import com.vungle.ads.TpatError
import com.vungle.ads.internal.Constants
import com.vungle.ads.internal.model.ErrorInfo
import com.vungle.ads.internal.persistence.FilePreferences
import com.vungle.ads.internal.persistence.FilePreferences.Companion.GENERIC_TPAT_FAILED_FILENAME
import com.vungle.ads.internal.persistence.FilePreferences.Companion.TPAT_FAILED_FILENAME
import com.vungle.ads.internal.protos.Sdk.SDKError
import com.vungle.ads.internal.signals.SignalManager
import com.vungle.ads.internal.util.LogEntry
import com.vungle.ads.internal.util.PathProvider
import kotlinx.serialization.Serializable
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import java.util.concurrent.Executor
import java.util.regex.Pattern

@Serializable
enum class HttpMethod {
    GET, POST
}

@Serializable
data class GenericTpatRequest(
    val method: HttpMethod = HttpMethod.GET,
    val headers: Map<String, String>? = null,
    val body: String? = null,
    var attempt: Int = 0
)

class TpatSender(
    val vungleApiClient: VungleApiClient,
    val logEntry: LogEntry? = null,
    ioExecutor: Executor,
    pathProvider: PathProvider,
    val signalManager: SignalManager? = null
) {
    companion object {
        private const val TAG: String = "TpatSender"
        private const val FAILED_TPATS = "FAILED_TPATS"
        private const val FAILED_GENERIC_TPATS = "FAILED_GENERIC_TPATS"
        private const val MAX_RETRIES = 5
    }

    private val tpatFilePreferences =
        FilePreferences.get(ioExecutor, pathProvider, TPAT_FAILED_FILENAME)
    private val genericTpatFilePreferences =
        FilePreferences.get(ioExecutor, pathProvider, GENERIC_TPAT_FAILED_FILENAME)

    fun sendWinNotification(urlString: String, executor: Executor) {
        val url = injectSessionIdToUrl(urlString)
        executor.execute {
            val error = vungleApiClient.pingTPAT(url, logEntry = logEntry)
            if (error != null) {
                TpatError(
                    SDKError.Reason.AD_WIN_NOTIFICATION_ERROR,
                    "Fail to send $url, error: ${error.description}"
                ).setLogEntry(logEntry).logErrorNoReturnValue()
            }
        }
    }

    fun sendTpats(urls: Iterable<String>, executor: Executor) {
        urls.forEach { sendTpat(it, executor) }
    }

    fun sendTpat(url: String, executor: Executor) {
        val urlWithSessionId = injectSessionIdToUrl(url)
        executor.execute {
            val storedTpats = getStoredTpats()
            val attemptNumber = storedTpats[url] ?: 0
            val error = vungleApiClient.pingTPAT(urlWithSessionId, logEntry = logEntry)
            if (error == null) {
                if (attemptNumber != 0) {
                    storedTpats.remove(url)
                    saveStoredTpats(storedTpats)
                }
            } else {
                if (!error.errorIsTerminal) {
                    if (attemptNumber >= MAX_RETRIES) {
                        storedTpats.remove(url)
                        saveStoredTpats(storedTpats)
                        TpatError(
                            SDKError.Reason.TPAT_RETRY_FAILED,
                            urlWithSessionId
                        ).setLogEntry(logEntry).logErrorNoReturnValue()
                    } else {
                        storedTpats[url] = attemptNumber + 1
                        saveStoredTpats(storedTpats)
                    }
                }
                logTpatError(error, urlWithSessionId)
            }
        }
    }

    fun sendGenericTpat(
        url: String,
        request: GenericTpatRequest,
        retry: Boolean,
        executor: Executor
    ) {
        val urlWithSessionId = injectSessionIdToUrl(url)
        executor.execute {
            val storedTpats = getStoredGenericTpats()
            val attemptNumber = storedTpats[url]?.attempt ?: 0

            val error = when (request.method) {
                HttpMethod.GET -> vungleApiClient.pingTPAT(
                    urlWithSessionId, request.headers, logEntry = logEntry
                )

                HttpMethod.POST -> vungleApiClient.pingTPAT(
                    urlWithSessionId,
                    request.headers,
                    request.body,
                    HttpMethod.POST,
                    logEntry
                )
            }

            if (error == null) {
                if (attemptNumber != 0) {
                    storedTpats.remove(url)
                    saveStoredGenericTpats(storedTpats)
                }
            } else {
                if (!error.errorIsTerminal && retry) {
                    if (attemptNumber >= MAX_RETRIES) {
                        storedTpats.remove(url)
                        saveStoredGenericTpats(storedTpats)
                        TpatError(
                            SDKError.Reason.TPAT_RETRY_FAILED,
                            urlWithSessionId
                        ).setLogEntry(logEntry).logErrorNoReturnValue()
                    } else {
                        val genericTpatRequest = storedTpats[url]?.copy(attempt = attemptNumber + 1)
                        storedTpats[url] = genericTpatRequest ?: GenericTpatRequest(
                            method = request.method,
                            headers = request.headers,
                            body = request.body,
                            attempt = attemptNumber + 1
                        )
                        saveStoredGenericTpats(storedTpats)
                    }
                }
                logTpatError(error, urlWithSessionId)
            }
        }
    }

    /**
     * Retrieves a map of TPAT failures stored in shared preferences.
     * The TPAT events are hardcoded in the ad response.
     *
     * @return A `Map` where the key is the TPAT event and the value is the retry attempt.
     * If no stored TPAT failures are found or if there's an error in decoding the stored TPAT failures,
     * an empty `Map` is returned.
     */
    private fun getStoredTpats(): MutableMap<String, Int> {
        val storedTpats = tpatFilePreferences.getString(FAILED_TPATS)
        return storedTpats?.let {
            runCatching { Json.decodeFromString<MutableMap<String, Int>>(it) }.onFailure {
                Logger.e(TAG, "Failed to decode stored tpats: $it")
            }.getOrNull() ?: mutableMapOf()
        } ?: mutableMapOf()
    }

    /**
     * Retrieves a map of generic TPAT failures stored in shared preferences.
     * The generic TPAT events are triggered by template and not hardcoded in the ad response.
     *
     * @return A `Map` where the key is the TPAT event and the value is the `GenericTpatRequest` object.
     * If no stored generic TPAT failures are found or if there's an error in decoding the stored generic TPAT failures,
     * an empty `Map` is returned.
     */
    private fun getStoredGenericTpats(): MutableMap<String, GenericTpatRequest> {
        val storedTpats = genericTpatFilePreferences.getString(FAILED_GENERIC_TPATS)
        return storedTpats?.let {
            runCatching { Json.decodeFromString<MutableMap<String, GenericTpatRequest>>(it) }.onFailure {
                Logger.e(TAG, "Failed to decode stored generic tpats: $it")
            }.getOrNull() ?: mutableMapOf()
        } ?: mutableMapOf()
    }

    /**
     * Stores a map of TPAT failures into shared preferences.
     * The TPAT events are hardcoded in the ad response.
     *
     * @param tpats A `Map` where the key is the TPAT event and the value is the retry attempt.
     * If there's an error in encoding the TPAT failures for storage, an error message is logged.
     */
    private fun saveStoredTpats(tpats: MutableMap<String, Int>) {
        runCatching {
            tpatFilePreferences
                .put(FAILED_TPATS, Json.encodeToString(tpats))
                .apply()
        }.onFailure { e ->
            Logger.e(TAG, "Failed to encode the about to storing tpats: $tpats")
        }
    }

    /**
     * Stores a map of generic TPAT failures into shared preferences.
     * The generic TPAT events are triggered by template and not hardcoded in the ad response.
     *
     * @param tpats A `Map` where the key is the TPAT event and the value is the `GenericTpatRequest` object.
     * If there's an error in encoding the generic TPAT failures for storage, an error message is logged.
     */
    private fun saveStoredGenericTpats(tpats: MutableMap<String, GenericTpatRequest>) {
        runCatching {
            genericTpatFilePreferences
                .put(FAILED_GENERIC_TPATS, Json.encodeToString(tpats))
                .apply()
        }.onFailure { e ->
            Logger.e(TAG, "Failed to encode the about to storing generic tpats: $tpats")
        }
    }

    private fun logTpatError(
        error: ErrorInfo,
        urlWithSessionId: String
    ) {
        Logger.e(TAG, "Failed with ${error.description}, url:$urlWithSessionId")
        TpatError(
            error.reason,
            "Fail to send $urlWithSessionId, error: ${error.description}"
        ).setLogEntry(logEntry).logErrorNoReturnValue()
    }

    internal fun resendStoredTpats(executor: Executor) {
        getStoredTpats().forEach { (url, _) -> sendTpat(url, executor) }
        getStoredGenericTpats().forEach { (url, request) ->
            sendGenericTpat(
                url,
                GenericTpatRequest(
                    method = request.method,
                    headers = request.headers,
                    body = request.body
                ),
                retry = true,
                executor
            )
        }
    }

    @VisibleForTesting
    fun injectSessionIdToUrl(url: String): String {
        val sessionId = signalManager?.uuid ?: ""
        return if (sessionId.isNotEmpty()) {
            url.replace(
                Pattern.quote(Constants.SESSION_ID).toRegex(), sessionId
            )
        } else {
            url
        }
    }
}
