package com.vungle.ads.internal.network

import androidx.annotation.VisibleForTesting
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.model.CommonRequestBody
import com.vungle.ads.internal.model.ConfigPayload
import com.vungle.ads.internal.network.converters.EmptyResponseConverter
import com.vungle.ads.internal.network.converters.JsonConverter
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import okhttp3.Headers.Companion.toHeaders
import okhttp3.HttpUrl
import okhttp3.HttpUrl.Companion.toHttpUrl
import okhttp3.Request
import okhttp3.RequestBody
import okhttp3.RequestBody.Companion.toRequestBody
import kotlin.reflect.typeOf

class VungleApiImpl(@get:VisibleForTesting internal val okHttpClient: okhttp3.Call.Factory) :
    VungleApi {

    companion object {
        private val json = Json {
            ignoreUnknownKeys = true
            encodeDefaults = true
            explicitNulls = false
            allowStructuredMapKeys = true
        }
        private const val VUNGLE_VERSION = "7.1.0"
    }

    private var appId: String? = null

    private val emptyResponseConverter = EmptyResponseConverter()

    override fun setAppId(appId: String) {
        this.appId = appId
    }

    private fun defaultBuilder(ua: String, path: String, placementReferenceId: String? = null, headers: Map<String, String>? = null): Request.Builder {
        val builder = Request.Builder()
            .url(path)
            .addHeader("User-Agent", ua)
            .addHeader("Vungle-Version", VUNGLE_VERSION)
            .addHeader("Content-Type", "application/json")
        appId?.let {
            builder.addHeader("X-Vungle-App-Id", it)
        }
        headers?.let { builder.headers(it.toHeaders()) }

        placementReferenceId?.let {
            builder.addHeader("X-Vungle-Placement-Ref-Id", it)
        }

        return builder
    }

    private fun defaultProtoBufBuilder(ua: String, path: String): Request.Builder {
        val builder = Request.Builder()
            .url(path)
            .addHeader("User-Agent", ua)
            .addHeader("Vungle-Version", VUNGLE_VERSION)
            .addHeader("Content-Type", "application/x-protobuf")
        appId?.let {
            builder.addHeader("X-Vungle-App-Id", it)
        }

        return builder
    }

    override fun config(ua: String, path: String, body: CommonRequestBody): Call<ConfigPayload>? {
        return try {
            val requestBody = json.encodeToString(body)

            val request = defaultBuilder(ua, path)
                .post(requestBody.toRequestBody(null))
                .build()

            OkHttpCall(okHttpClient.newCall(request), JsonConverter(typeOf<ConfigPayload>()))
        } catch (e: Exception) {
            null
        }
    }

    override fun ads(ua: String, path: String, body: CommonRequestBody): Call<AdPayload>? {
        return try {
            val requestBody = json.encodeToString(body)
            val placementReferenceId = body.request?.placements?.firstOrNull()

            val request = defaultBuilder(ua, path, placementReferenceId)
                .post(requestBody.toRequestBody(null))
                .build()

            OkHttpCall(okHttpClient.newCall(request), JsonConverter(typeOf<AdPayload>()))
        } catch (e: Exception) {
            null
        }
    }

    override fun ri(ua: String, path: String, body: CommonRequestBody): Call<Void>? {
        return try {
            val requestBody = json.encodeToString(body)

            val request = defaultBuilder(ua, path)
                .post(requestBody.toRequestBody(null))
                .build()

            OkHttpCall(okHttpClient.newCall(request), emptyResponseConverter)
        } catch (e: Exception) {
            null
        }
    }

    override fun pingTPAT(
        ua: String,
        url: String,
        requestType: HttpMethod,
        headers: Map<String, String>?,
        requestBody: RequestBody?
    ): Call<Void> {
        val urlBuilder: HttpUrl.Builder = url.toHttpUrl().newBuilder()
        val requestBuilder =
            defaultBuilder(ua, urlBuilder.build().toString(), headers = headers)
        val request = when (requestType) {
            HttpMethod.GET -> {
                requestBuilder.get().build()
            }

            HttpMethod.POST -> {
                val body =
                    requestBody ?: ByteArray(0).toRequestBody(null)
                requestBuilder.post(body).build()
            }
        }
        return OkHttpCall(okHttpClient.newCall(request), emptyResponseConverter)
    }

    override fun sendMetrics(ua:String, path: String, requestBody: RequestBody): Call<Void> {
        val urlBuilder: HttpUrl.Builder = path.toHttpUrl().newBuilder()
        val request = defaultProtoBufBuilder(ua, urlBuilder.build().toString())
            .post(requestBody).build()

        return OkHttpCall(okHttpClient.newCall(request), emptyResponseConverter)
    }

    override fun sendErrors(ua: String, path: String, requestBody: RequestBody): Call<Void> {
        val urlBuilder: HttpUrl.Builder = path.toHttpUrl().newBuilder()
        val request = defaultProtoBufBuilder(ua, urlBuilder.build().toString())
            .post(requestBody).build()

        return OkHttpCall(okHttpClient.newCall(request), emptyResponseConverter)
    }

    override fun sendAdMarkup(path: String, requestBody: RequestBody): Call<Void> {
        val urlBuilder: HttpUrl.Builder = path.toHttpUrl().newBuilder()

        val request = defaultBuilder("debug", urlBuilder.build().toString())
            .post(requestBody)
            .build()
        return OkHttpCall(okHttpClient.newCall(request), emptyResponseConverter)
    }
}
