package com.vungle.ads.internal.platform

import android.content.Context
import android.media.AudioManager
import android.os.Build
import android.os.Environment
import android.os.PowerManager
import android.provider.Settings
import android.provider.Settings.SettingNotFoundException
import android.telephony.TelephonyManager
import androidx.core.util.Consumer
import com.google.android.gms.ads.identifier.AdvertisingIdClient
import com.google.android.gms.appset.AppSet
import com.google.android.gms.appset.AppSetIdClient
import com.google.android.gms.appset.AppSetIdInfo
import com.google.android.gms.common.GooglePlayServicesNotAvailableException
import com.google.android.gms.tasks.Task
import com.vungle.ads.internal.executor.VungleThreadPoolExecutor
import com.vungle.ads.internal.model.AdvertisingInfo
import com.vungle.ads.internal.platform.Platform.Companion.MANUFACTURER_AMAZON
import com.vungle.ads.internal.util.Logger


class AndroidPlatform(
    private val context: Context,
    private val uaExecutor: VungleThreadPoolExecutor,
) : Platform {
    init {
        updateAppSetID()
    }

    companion object {
        private const val TAG = "AndroidPlatform"

        internal fun getCarrierName(context: Context) =
            (context.getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager).networkOperatorName
    }

    private var appSetId: String? = null
    private var appSetIdScope: Int? = null
    private var advertisingInfo: AdvertisingInfo? = null
    private val powerManager: PowerManager =
        context.getSystemService(Context.POWER_SERVICE) as PowerManager
    private val audioManager: AudioManager =
        context.getSystemService(Context.AUDIO_SERVICE) as AudioManager

    override val isBatterySaverEnabled: Boolean
        get() = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            powerManager.isPowerSaveMode
        } else false
    override val isSideLoaded: Boolean = false // Do not retrieve canRequestPackageInstalls

    override val volumeLevel: Float
        get() = try {
            val max = audioManager.getStreamMaxVolume(AudioManager.STREAM_MUSIC).toFloat()
            val current = audioManager.getStreamVolume(AudioManager.STREAM_MUSIC).toFloat()
            current / max
        } catch (_: Exception) {
            0f
        }
    override val isSoundEnabled: Boolean
        get() = try {
            val current = audioManager.getStreamVolume(AudioManager.STREAM_MUSIC)
            current > 0
        } catch (_: Exception) {
            true
        }
    override val carrierName: String
        get() = getCarrierName(context)
    override val isSdCardPresent: Boolean
        get() = try {
            Environment.getExternalStorageState() == Environment.MEDIA_MOUNTED
        } catch (e: Exception) {
            Logger.e(TAG, "Acquiring external storage state failed", e)
            false
        }

    override fun getUserAgentLazy(consumer: Consumer<String?>) {
        uaExecutor.execute {
            WebViewUtil(context).getUserAgent(consumer)
        }
    }//Generally not possible but for safer side

    private fun getAmazonAdvertisingInfo(): AdvertisingInfo {
        val advertisingInfo = AdvertisingInfo()
        try {
            val cr = context.contentResolver
            advertisingInfo.limitAdTracking = Settings.Secure.getInt(cr, "limit_ad_tracking") == 1
            advertisingInfo.advertisingId = Settings.Secure.getString(cr, "advertising_id")

        } catch (ex: SettingNotFoundException) {
            Logger.w(TAG, "Error getting Amazon advertising info: Setting not found.", ex)
        } catch (ex: Exception) {
            Logger.w(TAG, "Error getting Amazon advertising info", ex)
        }

        return advertisingInfo
    }

    private fun getGoogleAdvertisingInfo(): AdvertisingInfo {
        val advertisingInfo = AdvertisingInfo()
        try {
            val idInfo = AdvertisingIdClient.getAdvertisingIdInfo(context)
            advertisingInfo.advertisingId = idInfo.id

            advertisingInfo.limitAdTracking = idInfo.isLimitAdTrackingEnabled
        } catch (ex: NoClassDefFoundError) {
            Logger.e(TAG, "Play services Not available: " + ex.localizedMessage)
            val cr = context.contentResolver
            advertisingInfo.advertisingId = Settings.Secure.getString(cr, "advertising_id")
        } catch (exception: GooglePlayServicesNotAvailableException) {
            Logger.e(TAG, "Play services Not available: ${exception.localizedMessage}")
        } catch (exception: Exception) {
            Logger.e(TAG, "Error getting Google advertising info: ${exception.localizedMessage}")
        }
        return advertisingInfo
    }

    override fun getAdvertisingInfo(): AdvertisingInfo {
        advertisingInfo?.let { cachedAdvInfo ->
            if (!cachedAdvInfo.advertisingId.isNullOrEmpty()) {
                return cachedAdvInfo
            }
        }

        val newAdvertisingInfo =
            if (Build.MANUFACTURER.equals(MANUFACTURER_AMAZON, ignoreCase = true)) {
                getAmazonAdvertisingInfo()
            } else {
                getGoogleAdvertisingInfo()
            }

        this.advertisingInfo = newAdvertisingInfo
        return newAdvertisingInfo
    }

    override fun getAppSetId(): String? {
        return appSetId
    }

    override fun getAppSetIdScope(): Int? {
        return appSetIdScope
    }

    // AppSetIdInfo#getId might potentially return null causing a NPE. Possibly an issue with
    // Google's implementation.
    private fun updateAppSetID() {

        //query value for AppSetId
        if (!appSetId.isNullOrEmpty()) return
        try {
            val client: AppSetIdClient = AppSet.getClient(context)
            val task: Task<AppSetIdInfo> = client.appSetIdInfo
            task.addOnSuccessListener { info ->
                if (info != null) {
                    // Read app set ID value, which uses version 4 of the
                    // universally unique identifier (UUID) format.
                    appSetId = info.id
                    appSetIdScope = info.scope
                }
            }
        } catch (ex: NoClassDefFoundError) {
            Logger.e(TAG, "Required libs to get AppSetID Not available: ${ex.localizedMessage}")
        } catch (ex: Exception) {
            Logger.e(TAG, "Error getting AppSetID: ${ex.localizedMessage}")
        }
    }

    //Very first time or after clearing app data
    override var userAgent: String? = null
        get() {
            return field ?: System.getProperty("http.agent")
        }
    override val isSilentModeEnabled: Boolean
        get() = try {
            audioManager.ringerMode == AudioManager.RINGER_MODE_SILENT ||
                    audioManager.ringerMode == AudioManager.RINGER_MODE_VIBRATE

        } catch (_: Exception) {
            false
        }

}