package com.vungle.ads.internal.util

import android.app.Activity
import android.app.Application
import android.content.Context
import android.content.Intent
import android.os.Bundle
import androidx.annotation.VisibleForTesting
import com.vungle.ads.internal.ui.PresenterAdOpenCallback
import java.lang.ref.WeakReference
import java.util.concurrent.CopyOnWriteArraySet
import java.util.concurrent.atomic.AtomicBoolean

internal class ActivityManager private constructor() : Application.ActivityLifecycleCallbacks {
    open class LifeCycleCallback {

        open fun onForeground() {
            //no-op
        }

        open fun onBackground() {
            //no-op
        }
    }

    private var isInitialized = AtomicBoolean(false)

    private var targetActivityInfo: TargetActivityInfo? = null

    private val startedActivities = CopyOnWriteArraySet<String>()

    /** if init + loadAd/playAd happens on same activity, there will be no transitions that
     * ActivityManager will be aware of, so if there were no transitions we have to guess that
     * app is in foreground. For the most part it will be true, but it is still guessing */
    private var lastStoppedActivityName : String? = null

    private val callbacks = CopyOnWriteArraySet<LifeCycleCallback>()

    private val isAppForeground
        get() = startedActivities.isNotEmpty()

    private val inBackground = AtomicBoolean(false)

    private fun init(context: Context) {
        if (this.isInitialized.getAndSet(true)) return
        try {
            ThreadUtil.runOnUiThread {
                val app = context.applicationContext as Application
                app.registerActivityLifecycleCallbacks(this)
            }
        } catch (e: Exception) {
            Logger.e(TAG, "Error initializing ActivityManager", e)
            isInitialized.set(false)
        }
    }

    internal fun deInit(context: Context) {
        val app = context.applicationContext as Application
        app.unregisterActivityLifecycleCallbacks(this)
        startedActivities.clear()
        this.isInitialized.set(false)
        callbacks.clear()
    }

    // Check current app foreground state
    private fun isAppInForeground(): Boolean {
        return !isInitialized.get() /* before init we don't know state, so just assume foreground */
                || lastStoppedActivityName == null /* assuming init is done in foreground, so app is
                 in foreground since init but before first onStop */
                || isAppForeground
    }

    fun addListener(callback: LifeCycleCallback) {
        callbacks.add(callback)
    }

    override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {
        //no-op
    }

    override fun onActivityStarted(activity: Activity) {
        startedActivities.add(activity.toString())
        if (!inBackground.getAndSet(false)) return

        for (callback in callbacks) {
            callback.onForeground()
        }

        targetActivityInfo?.let {
            val context = it.context.get()
            if (context != null) {
                startWhenForeground(
                    context,
                    it.deepLinkOverrideIntent,
                    it.defaultIntent,
                    it.adOpenCallback
                )
            }
            targetActivityInfo = null
        }
    }

    override fun onActivityStopped(activity: Activity) {
        lastStoppedActivityName = activity.toString()
        startedActivities.remove(activity.toString())
        val foreground = isAppInForeground()
        inBackground.set(!foreground)
        if (foreground) return

        for (callback in callbacks) {
            callback.onBackground()
        }
    }

    override fun onActivityResumed(activity: Activity) {
        //no-op
    }

    override fun onActivityPaused(activity: Activity) {
        //no-op
    }

    override fun onActivitySaveInstanceState(activity: Activity, outState: Bundle) {
        //no-op
    }

    override fun onActivityDestroyed(activity: Activity) {
        //no-op
    }

    private data class TargetActivityInfo(
        val context: WeakReference<Context>,
        val deepLinkOverrideIntent: Intent?,
        val defaultIntent: Intent?,
        val adOpenCallback: PresenterAdOpenCallback?
    )

    companion object {
        private const val TAG = "ActivityManager"

        @VisibleForTesting
        internal val instance = ActivityManager()

        fun startWhenForeground(
            context: Context,
            deepLinkOverrideIntent: Intent?,
            defaultIntent: Intent?,
            adOpenCallback: PresenterAdOpenCallback?
        ): Boolean {
            return if (isForeground()) {
                startActivityHandleException(
                    context,
                    deepLinkOverrideIntent,
                    defaultIntent,
                    adOpenCallback
                )
            } else {
                // API 29 and 30 are not work as expected as what google API said, so we need to start
                // activity in onActivityStarted manually
                instance.targetActivityInfo = TargetActivityInfo(
                    WeakReference(context),
                    deepLinkOverrideIntent,
                    defaultIntent,
                    adOpenCallback
                )
                return false
            }
        }

        private fun startActivityHandleException(
            context: Context,
            deepLinkOverrideIntent: Intent?,
            defaultIntent: Intent?,
            adOpenCallback: PresenterAdOpenCallback?
        ): Boolean {
            if (deepLinkOverrideIntent == null && defaultIntent == null) {
                return false
            }
            try {
                if (deepLinkOverrideIntent != null) {
                    context.startActivity(deepLinkOverrideIntent)

                    adOpenCallback?.onDeeplinkClick(true)
                } else {
                    context.startActivity(defaultIntent)
                }
            } catch (exception: Exception) {
                Logger.e(
                    TAG,
                    "Cannot launch/find activity to handle the Implicit intent: $exception"
                )
                try {
                    if (deepLinkOverrideIntent != null) {
                        adOpenCallback?.onDeeplinkClick(false)
                    }

                    if (deepLinkOverrideIntent == null || defaultIntent == null) {
                        return false
                    }
                    context.startActivity(defaultIntent)
                } catch (exception: Exception) {
                    return false
                }
            }

            return true
        }

        fun init(context: Context) {
            instance.init(context)
        }

        fun isForeground(): Boolean {
            val foreground = instance.isAppInForeground()
            return foreground
        }

        fun addLifecycleListener(listener: LifeCycleCallback) {
            instance.addListener(listener)
        }

        fun deInit(context: Context) {
            instance.deInit(context)
        }
    }
}
