@file:JvmName("Logger")

package com.vungle.ads.internal.util

import android.util.Log
import com.vungle.ads.BuildConfig
import java.util.regex.Pattern

/**
 * Use this class to do all Logging instead of calling Log.d, Log.e or Log.w
 */
class Logger {

    companion object {
        private var enabled: Boolean = BuildConfig.DEBUG

        fun enable(enabled: Boolean) {
            Companion.enabled = enabled || BuildConfig.DEBUG
        }

        @JvmStatic
        fun d(tag: String, message: String): Int {
            if (!enabled) {
                return -1
            }
            return Log.d(tag, message.eraseSensitiveData())
        }

        @JvmStatic
        fun e(tag: String, message: String): Int {
            if (!enabled) {
                return -1
            }
            return Log.e(tag, message.eraseSensitiveData())
        }

        @JvmStatic
        fun e(tag: String, message: String, throwable: Throwable): Int {
            if (!enabled) {
                return -1
            }
            return if (BuildConfig.DEBUG) {
                Log.e(tag, message.eraseSensitiveData(), throwable)
            } else {
                Log.e(
                    tag,
                    "${message.eraseSensitiveData()}; error: ${throwable.localizedMessage}"
                )
            }
        }

        @JvmStatic
        fun w(tag: String, message: String): Int {
            if (!enabled) {
                return -1
            }
            return Log.w(tag, message.eraseSensitiveData())
        }

        @JvmStatic
        fun w(tag: String, message: String, throwable: Throwable): Int {
            if (!enabled) {
                return -1
            }
            return if (BuildConfig.DEBUG) {
                Log.w(tag, message.eraseSensitiveData(), throwable)
            } else {
                Log.w(
                    tag,
                    "${message.eraseSensitiveData()}; error: ${throwable.localizedMessage}"
                )
            }
        }

        @JvmStatic
        fun i(tag: String, message: String): Int {
            if (!enabled) {
                return -1
            }
            return Log.i(tag, message.eraseSensitiveData())
        }

        @JvmStatic
        fun i(tag: String, message: String, throwable: Throwable): Int {
            if (!enabled) {
                return -1
            }
            return if (BuildConfig.DEBUG) {
                Log.i(tag, message.eraseSensitiveData(), throwable)
            } else {
                Log.i(
                    tag,
                    "${message.eraseSensitiveData()}; error: ${throwable.localizedMessage}"
                )
            }
        }

        fun String.eraseSensitiveData(): String {
            return this.replace(
                Pattern.compile("[\\d]{1,3}\\.[\\d]{1,3}\\.[\\d]{1,3}\\.[\\d]{1,3}").toRegex(),
                "xxx.xxx.xxx.xxx"
            )
        }


    }

}

class LogEntry {
    internal var placementRefId: String? = null
    internal var creativeId: String? = null
    internal var eventId: String? = null
    internal var adSource: String? = null

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (javaClass != other?.javaClass) return false

        other as LogEntry

        if (placementRefId != other.placementRefId) return false
        if (creativeId != other.creativeId) return false
        if (eventId != other.eventId) return false
        if (adSource != other.adSource) return false

        return true
    }

    override fun hashCode(): Int {
        var result = placementRefId.hashCode()
        result = 31 * result + creativeId.hashCode()
        result = 31 * result + eventId.hashCode()
        result = 31 * result + adSource.hashCode()
        return result
    }

    override fun toString(): String {
        return "LogEntry(placementRefId=$placementRefId, creativeId=$creativeId, eventId=$eventId, adSource=$adSource)"
    }
}
