package com.vungle.ads

import android.content.Context
import androidx.annotation.VisibleForTesting
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.internal.AdInternal
import com.vungle.ads.internal.load.AdLoaderCallback
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.signals.SignalManager
import com.vungle.ads.internal.signals.SignaledAd
import com.vungle.ads.internal.util.LogEntry
import com.vungle.ads.internal.util.ThreadUtil

/**
 * This class has attributes that we would want to share with Publishers for
 * any Ad that they have requested for, and all relevant info related to  the
 * Ad available in response to the request made
 */
abstract class BaseAd(
    val context: Context,
    val placementId: String,
    val adConfig: AdConfig
) : Ad {

    var adListener: BaseAdListener? = null
    internal val adInternal: AdInternal by lazy {
        constructAdInternal(context).also {
            it.logEntry = logEntry
        }
    }
    internal val signalManager: SignalManager by inject(context)

    internal val logEntry = LogEntry().also { it.placementRefId = placementId }

    /* Metrics */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal val requestToResponseMetric: TimeIntervalMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.AD_REQUEST_TO_RESPONSE_DURATION_MS)
    internal val responseToShowMetric: TimeIntervalMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.AD_RESPONSE_TO_SHOW_DURATION_MS)
    internal val presentToDisplayMetric: TimeIntervalMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.AD_PRESENT_TO_DISPLAY_DURATION_MS)
    internal val showToFailMetric: TimeIntervalMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.AD_SHOW_TO_FAIL_DURATION_MS)
    internal val displayToClickMetric: TimeIntervalMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.AD_DISPLAY_TO_CLICK_DURATION_MS)
    internal val leaveApplicationMetric: SingleValueMetric =
        SingleValueMetric(Sdk.SDKMetric.SDKMetricType.AD_LEAVE_APPLICATION)
    internal val rewardedMetric: SingleValueMetric =
        SingleValueMetric(Sdk.SDKMetric.SDKMetricType.AD_REWARD_USER)
    internal val showToCloseMetric: TimeIntervalMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.AD_SHOW_TO_CLOSE_DURATION_MS)

    // Generated by calling the fullscreen load method
    internal var signaledAd: SignaledAd? = null

    internal abstract fun constructAdInternal(context: Context): AdInternal

    override fun canPlayAd(): Boolean {
        return adInternal.canPlayAd() == null
    }

    var creativeId: String? = null
        private set
    var eventId: String? = null
        private set

    override fun load(adMarkup: String?) {
        requestToResponseMetric.markStart()
        adInternal.loadAd(
            placementId, adMarkup,
            object : AdLoaderCallback {
                override fun onSuccess(advertisement: AdPayload) {
                    onAdLoaded(advertisement)
                    onLoadSuccess(this@BaseAd, adMarkup)
                }

                override fun onFailure(error: VungleError) {
                    onLoadFailure(this@BaseAd, error)
                }
            }
        )
    }

    internal open fun onAdLoaded(advertisement: AdPayload) {
        advertisement.adConfig = adConfig
        creativeId = advertisement.getCreativeId()
        eventId = advertisement.eventId()
        signaledAd?.eventId = eventId
    }

    internal open fun onLoadSuccess(baseAd: BaseAd, adMarkup: String?) {
        ThreadUtil.runOnUiThread {
            adListener?.onAdLoaded(this)
        }
        onLoadEnd()
    }

    internal open fun onLoadFailure(baseAd: BaseAd, vungleError: VungleError) {
        ThreadUtil.runOnUiThread {
            adListener?.onAdFailedToLoad(this, vungleError)
        }
        onLoadEnd()
    }

    /* Both success and failure trigger this call */
    private fun onLoadEnd() {
        requestToResponseMetric.markEnd()
        AnalyticsClient.logMetric(requestToResponseMetric, logEntry)
        responseToShowMetric.markStart()
    }
}

interface Ad {
    fun canPlayAd(): Boolean?
    fun load(adMarkup: String? = null)
}

interface FullscreenAd : Ad {
    fun play(context: Context? = null)
}
