package com.vungle.ads.fpd

import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

@Serializable
class Demographic {

    @SerialName("age_range")
    private var ageRange: Int? = null

    @SerialName("length_of_residence")
    private var lengthOfResidence: Int? = null

    @SerialName("median_home_value_usd")
    private var medianHomeValueUSD: Int? = null

    @SerialName("monthly_housing_payment_usd")
    private var monthlyHousingPaymentUSD: Int? = null

    fun setAgeRange(age: Int): Demographic {
        this.ageRange = AgeRange.fromAge(age).id
        return this
    }

    fun setLengthOfResidence(lengthOfResidence: Int): Demographic {
        this.lengthOfResidence = LengthOfResidence.fromYears(lengthOfResidence).id
        return this
    }

    fun setMedianHomeValueUSD(homeValue: Int): Demographic {
        this.medianHomeValueUSD = MedianHomeValueUSD.fromPrice(homeValue).id
        return this
    }

    fun setMonthlyHousingCosts(housingCost: Int): Demographic {
        this.monthlyHousingPaymentUSD = MonthlyHousingCosts.fromCost(housingCost).id
        return this
    }
}

enum class AgeRange(val id: Int, val range: IntRange) {
    AGE_18_20(1, 18..20),
    AGE_21_30(2, 21..30),
    AGE_31_40(3, 31..40),
    AGE_41_50(4, 41..50),
    AGE_51_60(5, 51..60),
    AGE_61_70(6, 61..70),
    AGE_71_75(7, 71..75),
    OTHERS(0, Int.MIN_VALUE..Int.MAX_VALUE);  // Default range for others

    companion object {
        internal fun fromAge(age: Int): AgeRange {
            return values().firstOrNull { age in it.range } ?: OTHERS
        }
    }
}

enum class MonthlyHousingCosts(val id: Int, val range: IntRange) {
    UNDER_500(0, Int.MIN_VALUE..500),
    FROM_500_TO_1000(1, 501..1000),
    FROM_1000_TO_1500(2, 1001..1500),
    FROM_1500_TO_2000(3, 1501..2000),
    FROM_2000_TO_2500(4, 2001..2500),
    FROM_2500_TO_3000(5, 2501..3000),
    FROM_3000_TO_3500(6, 3001..3500),
    FROM_3500_TO_4000(7, 3501..4000),
    FROM_4000_TO_4500(8, 4001..4500),
    OVER_4500(9, 4501..Int.MAX_VALUE);

    companion object {
        internal fun fromCost(cost: Int): MonthlyHousingCosts {
            return values().firstOrNull { cost in it.range } ?: UNDER_500
        }
    }
}

enum class MedianHomeValueUSD(val id: Int, val range: IntRange) {
    UNDER_100K(0, Int.MIN_VALUE..100000),
    FROM_100K_TO_300K(1, 100001..300000),
    FROM_300K_TO_500K(2, 300001..500000),
    FROM_500K_TO_700K(3, 500001..700000),
    FROM_700K_TO_900K(4, 700001..900000),
    FROM_900K_TO_1M1(5, 900001..1100000),
    FROM_1M1_TO_1M3(6, 1100001..1300000),
    FROM_1M3_TO_1M5(7, 1300001..1500000),
    FROM_1M5_TO_1M7(8, 1500001..1700000),
    OVER_1M7(9, 1700001..Int.MAX_VALUE);

    companion object {
        internal fun fromPrice(price: Int): MedianHomeValueUSD {
            return values().firstOrNull { price in it.range } ?: UNDER_100K
        }
    }
}

//"description": "Duration of residence at the current location.
// 0=<1 year, 1=1-5 years, 2=6-10 years, 3=11-20 years, 4=21-30 years,
// 5=31-40 years, 6=41-50 years, 7=51-60 years, 8=61-70 years, 9=71-75+ years",
enum class LengthOfResidence(val id: Int, val range: IntRange) {
    LESS_THAN_ONE_YEAR(0, Int.MIN_VALUE..0),
    ONE_TO_FIVE_YEARS(1, 1..5),
    SIX_TO_TEN_YEARS(2, 6..10),
    ELEVEN_TO_TWENTY_YEARS(3, 11..20),
    TWENTY_ONE_TO_THIRTY_YEARS(4, 21..30),
    THIRTY_ONE_TO_FORTY_YEARS(5, 31..40),
    FORTY_ONE_TO_FIFTY_YEARS(6, 41..50),
    FIFTY_ONE_TO_SIXTY_YEARS(7, 51..60),
    SIXTY_ONE_TO_SEVENTY_YEARS(8, 61..70),
    OVER_SEVENTY_ONE_YEARS(9, 71..Int.MAX_VALUE);

    companion object {
        internal fun fromYears(years: Int): LengthOfResidence {
            return values().firstOrNull { years in it.range } ?: LESS_THAN_ONE_YEAR
        }
    }
}

