package com.vungle.ads.internal.load

import android.content.Context
import com.vungle.ads.APIFailedStatusCodeError
import com.vungle.ads.AdResponseEmptyError
import com.vungle.ads.AdRetryActiveError
import com.vungle.ads.AdRetryError
import com.vungle.ads.NetworkTimeoutError
import com.vungle.ads.NetworkUnreachable
import com.vungle.ads.SingleValueMetric
import com.vungle.ads.VungleAdSize
import com.vungle.ads.VungleError
import com.vungle.ads.internal.downloader.Downloader
import com.vungle.ads.internal.executor.Executors
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.model.Placement
import com.vungle.ads.internal.network.Call
import com.vungle.ads.internal.network.Callback
import com.vungle.ads.internal.network.Response
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.omsdk.OMInjector
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.util.PathProvider
import java.net.SocketTimeoutException

class DefaultAdLoader(
    context: Context,
    vungleApiClient: VungleApiClient,
    sdkExecutors: Executors,
    omInjector: OMInjector,
    downloader: Downloader,
    pathProvider: PathProvider,
    adRequest: AdRequest
) : BaseAdLoader(
    context, vungleApiClient, sdkExecutors, omInjector, downloader, pathProvider, adRequest
) {

    override fun requestAd() {
        fetchAdMetadata(adRequest.requestAdSize, adRequest.placement)
    }

    override fun onAdLoadReady() {
        // Do nothing.
    }

    private fun fetchAdMetadata(
        adSize: VungleAdSize?,
        placement: Placement
    ) {
        if (vungleApiClient.checkIsRetryAfterActive(placement.referenceId)) {
            onAdLoadFailed(AdRetryActiveError().setLogEntry(logEntry).logError())
            return
        }
        val adsCall = vungleApiClient.requestAd(placement.referenceId, adSize)
        if (adsCall == null) {
            onAdLoadFailed(
                NetworkUnreachable("adsCall is null").setLogEntry(logEntry).logError()
            )
            return
        }
        adsCall.enqueue(object : Callback<AdPayload> {
            override fun onResponse(call: Call<AdPayload>?, response: Response<AdPayload>?) {

                sdkExecutors.backgroundExecutor.execute {
                    if (vungleApiClient.getRetryAfterHeaderValue(placement.referenceId) > 0) {
                        onAdLoadFailed(AdRetryError().setLogEntry(logEntry).logError())
                        return@execute
                    }
                    if (response?.isSuccessful == false) {
                        onAdLoadFailed(
                            APIFailedStatusCodeError("ads API: ${response.code()}")
                                .setLogEntry(logEntry).logError()
                        )
                        return@execute
                    }
                    val adPayload = response?.body()
                    if (adPayload?.adUnit() == null) {
                        onAdLoadFailed(
                            AdResponseEmptyError("Ad response is empty")
                                .setLogEntry(logEntry).logError()
                        )
                        return@execute
                    }

                    handleAdMetaData(
                        adPayload,
                        SingleValueMetric(Sdk.SDKMetric.SDKMetricType.CONFIG_LOADED_FROM_AD_LOAD)
                    )
                }
            }

            override fun onFailure(call: Call<AdPayload>?, t: Throwable?) {
                sdkExecutors.backgroundExecutor.execute {
                    val error: VungleError = retrofitToVungleError(t)
                    onAdLoadFailed(error.setLogEntry(logEntry).logError())
                }
            }
        })

    }

    private fun retrofitToVungleError(throwable: Throwable?): VungleError {
        return when (throwable) {
            is SocketTimeoutException -> NetworkTimeoutError()
            else -> NetworkUnreachable("ads request fail: ${throwable?.message}")
        }
    }

}
